"""
SHAC Studio Transparent Sampler Bridge

Revolutionary principle: PRESERVE EVERYTHING.
Load audio exactly as recorded - no conversion chains, no molding.

If soundfile can't load it directly, we don't need to torture it through conversion.
Audio quality preservation > format support.
"""

import logging
import os
import sys
import time
import uuid
from pathlib import Path
from typing import Dict, List, Optional, Tuple
from tkinter import messagebox

# Set up logger
logger = logging.getLogger(__name__)

# Core audio loading - prioritize quality preservation
try:
    import soundfile as sf
    SOUNDFILE_AVAILABLE = True
    logger.info("soundfile available - high quality audio loading")
except ImportError:
    SOUNDFILE_AVAILABLE = False
    logger.warning("soundfile not available - limited format support")

try:
    import librosa
    LIBROSA_AVAILABLE = True
    logger.info("librosa available - fallback audio loading")
except ImportError:
    LIBROSA_AVAILABLE = False
    logger.warning("librosa not available - very limited format support")

import numpy as np

# State management and SHAC export
sys.path.append(str(Path(__file__).parent.parent))
from core.state_manager import StatefulComponent, StateEvent, state_manager
from core.shac_export import save_audio_as_shac


class TransparentSamplerBridge(StatefulComponent):
    """
    Transparent audio sampler that preserves original quality completely.
    
    Core principle: If we can't load it cleanly, we tell the user.
    No conversion chains, no molding, no format torture.
    """
    
    def __init__(self):
        super().__init__("TransparentSamplerBridge")

        # Loaded samples in original format
        self.loaded_samples = {}  # sample_id: original audio data

        # Quality preservation tracking
        self.load_history = []

        # Main window reference for coordination
        self.main_window = None

        # Subscribe to AUDIO_DATA_EDITED events for undo/redo support
        self.subscribe_to_state(StateEvent.AUDIO_DATA_EDITED, 'on_audio_data_edited')

        logger.info("Transparent Sampler Bridge initialized - quality preservation first")
        
        
    def _validate_audio_file(self, filepath: Path) -> Tuple[bool, Optional[str]]:
        """
        Validate audio file before attempting to load.

        Returns:
            (is_valid, error_message)
        """
        # Check file exists
        if not filepath.exists():
            return False, f"File not found: {filepath}"

        # Check file extension (supported formats)
        valid_extensions = {'.wav', '.mp3', '.flac', '.ogg', '.m4a', '.aiff', '.aif'}
        if filepath.suffix.lower() not in valid_extensions:
            return False, f"Unsupported audio format: {filepath.suffix}\nSupported: WAV, MP3, FLAC, OGG, M4A, AIFF"

        # Check file size (limit to 500MB to prevent memory exhaustion)
        MAX_FILE_SIZE = 500 * 1024 * 1024  # 500 MB
        file_size = filepath.stat().st_size
        if file_size > MAX_FILE_SIZE:
            size_mb = file_size / 1024 / 1024
            return False, f"File too large: {size_mb:.1f}MB (maximum: 500MB)\n\nConsider:\n• Trimming the audio\n• Converting to a compressed format\n• Splitting into multiple files"

        # Check file is readable
        try:
            with open(filepath, 'rb') as f:
                f.read(1024)  # Try to read first KB
        except PermissionError:
            return False, f"Permission denied: Cannot read {filepath.name}"
        except Exception as e:
            return False, f"Cannot read file: {e}"

        return True, None

    def _load_audio_file_internal(self, filepath: str) -> Tuple[bool, Dict]:
        """
        Internal method to load audio file preserving EVERYTHING.

        Priority: Quality preservation > format support
        If we can't load it cleanly, we tell the user honestly.
        """
        filepath = Path(filepath)

        # Validate file before attempting load
        is_valid, error_msg = self._validate_audio_file(filepath)
        if not is_valid:
            return False, {"error": error_msg}

        logger.info(f"Loading audio file: {filepath.name}")
        load_start = time.time()
        
        # Method 1: soundfile (cleanest, best quality preservation)
        if SOUNDFILE_AVAILABLE:
            try:
                # Load in natural format - no forced reshaping
                audio_data, sample_rate = sf.read(str(filepath))
                
                # Handle natural shape - preserve what the file actually is
                if audio_data.ndim == 1:
                    # Truly mono file - keep as 1D
                    channels = 1
                    logger.debug(f"Natural mono file: {audio_data.shape}")
                else:
                    # Multi-channel file - keep original shape
                    channels = audio_data.shape[1]
                    logger.debug(f"Natural {channels}-channel file: {audio_data.shape}")
                
                # Get complete format information
                info = sf.info(str(filepath))
                
                # Success - no conversion needed
                load_time = time.time() - load_start
                
                # Calculate duration
                duration = len(audio_data) / sample_rate

                # Validate duration (max 15 minutes to prevent memory issues)
                MAX_DURATION = 900  # 15 minutes in seconds
                if duration > MAX_DURATION:
                    minutes = duration / 60
                    return False, {"error": f"Audio file too long: {minutes:.1f} minutes (maximum: 15 minutes)\n\nFor longer audio:\n• Split into shorter segments\n• This prevents memory exhaustion\n• Spatial audio works best with shorter sources"}

                result = {
                    'audio_data': audio_data,
                    'sample_rate': sample_rate,
                    'channels': channels,
                    'bit_depth': info.subtype,
                    'format': info.format,
                    'duration': duration,
                    'frames': len(audio_data),
                    'filepath': str(filepath),
                    'load_method': 'soundfile_direct',
                    'load_time': load_time,
                    'quality_preserved': True,
                    'modifications': []
                }

                # Track quality preservation
                self.load_history.append({
                    'file': filepath.name,
                    'method': 'soundfile_direct',
                    'preserved': True,
                    'original_format': f"{sample_rate}Hz, {info.subtype}, {channels}ch",
                    'load_time': load_time
                })

                logger.info(f"Loaded preserving quality: {sample_rate}Hz, {info.subtype}, {channels}ch ({load_time:.3f}s)")
                return True, result
                
            except Exception as e:
                logger.warning(f"soundfile couldn't load {filepath.name}: {e}")

        # Method 2: librosa (some quality loss but still usable)
        if LIBROSA_AVAILABLE:
            try:
                logger.info(f"Trying librosa for {filepath.name}...")
                
                # Load without forcing parameters
                audio_data, sample_rate = librosa.load(str(filepath), sr=None, mono=False)
                
                # Handle librosa's format quirks
                if len(audio_data.shape) == 1:
                    audio_data = audio_data.reshape(-1, 1)
                else:
                    audio_data = audio_data.T  # librosa uses (channels, samples)
                    
                load_time = time.time() - load_start

                # Calculate duration
                duration = len(audio_data) / sample_rate

                # Validate duration (max 15 minutes)
                MAX_DURATION = 900  # 15 minutes in seconds
                if duration > MAX_DURATION:
                    minutes = duration / 60
                    return False, {"error": f"Audio file too long: {minutes:.1f} minutes (maximum: 15 minutes)\n\nFor longer audio:\n• Split into shorter segments\n• This prevents memory exhaustion\n• Spatial audio works best with shorter sources"}

                result = {
                    'audio_data': audio_data,
                    'sample_rate': sample_rate,
                    'channels': audio_data.shape[1],
                    'bit_depth': 'float32',  # librosa conversion
                    'format': 'converted',
                    'duration': duration,
                    'frames': len(audio_data),
                    'filepath': str(filepath),
                    'load_method': 'librosa',
                    'load_time': load_time,
                    'quality_preserved': False,
                    'modifications': ['Converted to float32 via librosa']
                }
                
                # Track quality note
                self.load_history.append({
                    'file': filepath.name,
                    'method': 'librosa',
                    'preserved': False,
                    'original_format': f"{sample_rate}Hz, float32, {audio_data.shape[1]}ch",
                    'load_time': load_time,
                    'note': 'Converted via librosa - some precision loss'
                })
                
                logger.warning(f"Loaded via librosa: {sample_rate}Hz, float32, {audio_data.shape[1]}ch ({load_time:.3f}s) - some precision loss")
                return True, result

            except Exception as e:
                logger.error(f"librosa couldn't load {filepath.name}: {e}")

        # No clean loading method available
        error_msg = f"Cannot load {filepath.name} while preserving quality"
        if not SOUNDFILE_AVAILABLE and not LIBROSA_AVAILABLE:
            error_msg += " - No audio libraries available"
        elif not SOUNDFILE_AVAILABLE:
            error_msg += " - Install soundfile for better format support"

        logger.error(error_msg)
        
        return False, {
            "error": error_msg,
            "file": str(filepath),
            "available_methods": {
                "soundfile": SOUNDFILE_AVAILABLE,
                "librosa": LIBROSA_AVAILABLE
            }
        }
        
    def add_audio_to_project(self, filepath: str, position: Tuple[float, float, float] = (0, 2, 0), 
                           source_id: str = None, name: str = None) -> str:
        """
        THE way to add audio to SHAC Studio.
        
        One method, complete workflow, perfect coordination.
        Loads audio ONCE, preserves quality, updates ALL components.
        
        Args:
            filepath: Path to audio file
            position: 3D position (x, y, z) in meters
            source_id: Optional specific ID to use (generates UUID if None)
            name: Optional display name (uses filename if None)
            
        Returns:
            source_id if successful, None if failed
        """
        # 1. Load audio ONCE, preserving quality
        success, audio_data = self._load_audio_file_internal(filepath)

        if not success:
            error_msg = audio_data.get('error', 'Unknown error')
            logger.error(f"Cannot add {Path(filepath).name}: {error_msg}")

            # Show user-friendly error dialog
            try:
                messagebox.showerror(
                    "Cannot Load Audio File",
                    f"Failed to load: {Path(filepath).name}\n\n{error_msg}"
                )
            except Exception:
                pass  # Headless mode, print already done

            return None
            
        # 2. Generate IDs and metadata
        source_id = source_id if source_id is not None else str(uuid.uuid4())
        name = name if name is not None else Path(filepath).stem
        
        # 3. Store in sampler bridge (source of truth)
        self.loaded_samples[source_id] = {
            **audio_data,
            'position': position,
            'name': name,
            'id': source_id,
            'filepath': str(filepath)
        }
        
        # 4. Update ALL components via single coordination point
        
        # Add to spatial view if available
        if hasattr(self, 'spatial_view') and self.spatial_view:
            self.spatial_view.add_source(source_id, name, position)
            
        # Add to audio engine preserving quality
        if hasattr(self, 'main_window') and self.main_window and hasattr(self.main_window, 'audio_engine'):
            engine_success = self.main_window.audio_engine.add_source(source_id, audio_data, position)
            if not engine_success:
                logger.warning(f"Audio engine couldn't add {source_id}")
        
        # 5. Emit state event for any listeners
        self.emit_state_change(StateEvent.SOURCE_ADDED, source_id, {
            'name': name,
            'filepath': str(filepath),
            'position': position,
            'sample_rate': audio_data['sample_rate'],
            'channels': audio_data['channels'],
            'duration': audio_data['duration'],
            'quality_preserved': audio_data.get('quality_preserved', False)
        })
        
        quality_note = "Quality preserved" if audio_data.get('quality_preserved', False) else "Some conversion applied"
        logger.info(f"Added '{name}' as '{source_id}' - {quality_note}")

        return source_id


    def set_spatial_view(self, spatial_view):
        """Set spatial view reference for positioning."""
        self.spatial_view = spatial_view
        logger.debug("Connected to spatial view")
        
    def get_sample_info(self, sample_id: str) -> Optional[Dict]:
        """Get complete sample information preserving original metadata."""
        return self.loaded_samples.get(sample_id)
        
    def get_audio_duration(self, sample_id: str) -> float:
        """Get audio duration in seconds - backward compatibility."""
        sample_info = self.loaded_samples.get(sample_id)
        if sample_info:
            return sample_info.get('duration', 0.0)
        return 0.0
        
    def get_audio_data(self, source_id: str, start_time: float = 0.0, duration: float = None) -> Optional[np.ndarray]:
        """
        Get audio data for a source - waveform panel compatibility.
        
        Args:
            source_id: The source identifier
            start_time: Start time in seconds (optional)
            duration: Duration in seconds (optional, None for full length)
            
        Returns:
            NumPy array with audio data or None if source not found
        """
        sample_info = self.loaded_samples.get(source_id)
        if not sample_info:
            return None
            
        audio_data = sample_info.get('audio_data')
        if audio_data is None:
            return None
            
        # Handle time-based extraction if requested
        if start_time > 0.0 or duration is not None:
            sample_rate = sample_info.get('sample_rate', 44100)
            start_frame = int(start_time * sample_rate)
            
            if duration is not None:
                end_frame = start_frame + int(duration * sample_rate)
                return audio_data[start_frame:end_frame]
            else:
                return audio_data[start_frame:]
        
        return audio_data
        
    def get_load_quality_report(self) -> List[Dict]:
        """Get report on how well we preserved audio quality during loading."""
        return self.load_history.copy()
        
    def get_supported_formats(self) -> List[str]:
        """Get list of formats we can load while preserving quality."""
        formats = []
        
        if SOUNDFILE_AVAILABLE:
            # soundfile supports these with high quality
            formats.extend(['.wav', '.flac', '.aiff', '.au', '.avr', '.caf', 
                           '.htk', '.svx', '.mat4', '.mat5', '.pvf', '.sds', 
                           '.sd2', '.voc', '.w64', '.xi', '.rf64'])
            
        if LIBROSA_AVAILABLE:
            # librosa adds these (with some quality loss)
            formats.extend(['.mp3', '.m4a', '.ogg', '.wma'])
            
        return sorted(list(set(formats)))
        
    def remove_sample(self, sample_id: str) -> bool:
        """Remove sample preserving any cached quality data."""
        if sample_id in self.loaded_samples:
            sample_info = self.loaded_samples[sample_id]
            
            # Remove from spatial view
            if hasattr(self, 'spatial_view') and self.spatial_view:
                self.spatial_view.remove_source(sample_id)
                
            # Remove from loaded samples
            del self.loaded_samples[sample_id]
            
            # Emit state change
            self.emit_state_change(StateEvent.SOURCE_REMOVED, sample_id, {
                'name': sample_info.get('name', sample_id)
            })

            logger.info(f"Removed '{sample_info.get('name', sample_id)}'")
            return True
            
        return False
        
    def remove_source(self, source_id: str) -> bool:
        """Alias for remove_sample - ensures compatibility across panels."""
        return self.remove_sample(source_id)
        
    def normalize_audio_levels(self, source_ids: List[str]):
        """Normalize audio levels across the specified sources."""
        if not source_ids:
            return

        logger.info(f"Normalizing audio levels for {len(source_ids)} sources")
        
        # Calculate RMS levels for all sources
        rms_levels = []
        for source_id in source_ids:
            if source_id in self.loaded_samples:
                audio_data = self.loaded_samples[source_id]['audio_data']
                rms = np.sqrt(np.mean(audio_data**2))
                rms_levels.append(rms)
            else:
                rms_levels.append(0.0)
        
        if not rms_levels or max(rms_levels) == 0:
            logger.warning("No audio data to normalize")
            return

        # Find target level (average RMS)
        target_rms = np.mean([rms for rms in rms_levels if rms > 0])

        # Apply normalization
        for i, source_id in enumerate(source_ids):
            if source_id in self.loaded_samples and rms_levels[i] > 0:
                current_rms = rms_levels[i]
                gain = target_rms / current_rms
                self.loaded_samples[source_id]['audio_data'] *= gain
                logger.debug(f"Normalized '{source_id}': gain={gain:.2f}")

        logger.info("Audio level normalization complete")
        
    def create_shac_composition(self, sources: List[Dict], filename: str, order: int,
                              sample_rate: int, listener_position: Tuple[float, float, float] = (0, 0, 0),
                              distance_gain: bool = False, length_handling: str = "pad",
                              progress_callback=None) -> bool:
        """Create a SHAC composition file from the provided sources."""
        try:
            logger.info(f"Creating SHAC composition: {filename}")
            logger.info(f"Settings: Order={order}, Sample Rate={sample_rate}Hz")
            logger.debug(f"Listener position: {listener_position}")
            logger.debug(f"Distance gain: {'ON' if distance_gain else 'OFF'}")
            logger.debug(f"Length handling: {length_handling}")
            
            # Prepare audio data dictionary using export names as keys
            audio_layers = {}
            positions = {}
            source_lengths = []
            export_name_to_source_id = {}  # Keep mapping for position updates

            for source_index, source in enumerate(sources):
                source_id = source['id']
                export_name = source.get('export_name', source_id)  # Use export name or fallback to ID
                
                if source_id in self.loaded_samples:
                    sample_info = self.loaded_samples[source_id]
                    audio_data = sample_info['audio_data']
                    original_sr = sample_info['sample_rate']

                    # Update progress - loading source
                    if progress_callback:
                        progress_callback(source_index, export_name, "Loading")

                    # Handle sample rate conversion if needed
                    if original_sr != sample_rate:
                        logger.info(f"Source '{export_name}': {original_sr}Hz -> {sample_rate}Hz")
                        if progress_callback:
                            progress_callback(source_index, export_name, "Resampling")
                        try:
                            if LIBROSA_AVAILABLE:
                                audio_data = librosa.resample(audio_data, orig_sr=original_sr, target_sr=sample_rate)
                                logger.debug(f"Resampled '{export_name}' successfully")
                            else:
                                logger.warning(f"Cannot resample '{export_name}' - librosa not available")
                                logger.info(f"Preserving original sample rate: {original_sr}Hz")
                                sample_rate = original_sr
                        except Exception as e:
                            logger.error(f"Resample failed for '{export_name}': {e}")
                            logger.info(f"Preserving original sample rate: {original_sr}Hz")
                            sample_rate = original_sr
                    
                    # Use export name as layer key (what the web player will see)
                    audio_layers[export_name] = audio_data
                    positions[export_name] = source['position']
                    export_name_to_source_id[export_name] = source_id
                    source_lengths.append(len(audio_data))
                    logger.debug(f"Added source '{export_name}' (was '{source_id}') at position {source['position']} ({len(audio_data)} samples)")

                    # Update progress - source processed
                    if progress_callback:
                        progress_callback(source_index, export_name, "Processed")
                else:
                    logger.warning(f"Source '{source_id}' not found in loaded samples")

            if not audio_layers:
                logger.error("No audio sources available for export")
                return False
            
            # Handle source length synchronization
            if len(source_lengths) > 1:
                min_length = min(source_lengths)
                max_length = max(source_lengths)
                
                if length_handling == "trim":
                    logger.info(f"Trimming all sources to {min_length} samples")
                    for export_name in audio_layers:
                        audio_layers[export_name] = audio_layers[export_name][:min_length]

                elif length_handling == "pad":
                    logger.info(f"Padding shorter sources to {max_length} samples")
                    for export_name in audio_layers:
                        audio_data = audio_layers[export_name]
                        if len(audio_data) < max_length:
                            padding = np.zeros(max_length - len(audio_data))
                            if audio_data.ndim == 2:  # stereo
                                padding = np.zeros((max_length - len(audio_data), audio_data.shape[1]))
                            audio_layers[export_name] = np.concatenate([audio_data, padding])
                            logger.debug(f"Padded '{export_name}' with {len(padding)} samples of silence")

                elif length_handling == "stretch":
                    logger.info(f"Time-stretching all sources to {max_length} samples")
                    if LIBROSA_AVAILABLE:
                        for export_name in audio_layers:
                            audio_data = audio_layers[export_name]
                            if len(audio_data) != max_length:
                                stretch_ratio = max_length / len(audio_data)
                                audio_layers[export_name] = librosa.effects.time_stretch(audio_data, rate=1/stretch_ratio)
                                logger.debug(f"Stretched '{export_name}' by {stretch_ratio:.2f}x")
                    else:
                        logger.warning("Time-stretching requires librosa - falling back to padding")
                        length_handling = "pad"  # fallback
                
            # Create metadata
            metadata = {
                'title': Path(filename).stem,
                'created_by': 'SHAC Studio',
                'listener_spawn_position': listener_position,
                'source_count': len(audio_layers),
                'creation_time': time.strftime('%Y-%m-%d %H:%M:%S')
            }
            
            # Export using SHAC codec
            output_path = save_audio_as_shac(
                audio_data=audio_layers,
                filename=filename,
                positions=positions,
                metadata=metadata,
                order=order,
                sample_rate=sample_rate,
                distance_gain=distance_gain
            )
            
            logger.info(f"SHAC composition saved: {output_path}")
            logger.debug(f"Layer names in SHAC file: {list(audio_layers.keys())}")
            return True

        except Exception as e:
            logger.error(f"SHAC export failed: {e}", exc_info=True)
            return False
            
    # LEGACY ZYZ FORMAT - NO LONGER SUPPORTED
    # ZYZ was an experimental pre-mixed format that has been deprecated
    # All exports now use the full SHAC format for maximum flexibility
    # This code is preserved for reference only
    """
    def create_zyz_composition(self, sources: List[Dict], filename: str,
                              order: int, sample_rate: int,
                              listener_position: Tuple[float, float, float] = (0, 0, 0),
                              distance_gain: bool = False,
                              progress_callback=None) -> bool:
        '''
        Create a ZYZ composition (pre-mixed single ambisonic field).

        ZYZ is 40x smaller than SHAC because all sources are summed into
        a single ambisonic soundfield instead of stored as separate layers.

        Preserves:
        - Full 3D spatial experience
        - WASD navigation
        - Ambisonic soundfield

        Loses:
        - Per-source manipulation
        - Source remixing
        - Import back to Studio

        Args:
            sources: List of source dicts with id, position, export_name
            filename: Output .zyz file path
            order: Ambisonic order (1, 3, 5, 7)
            sample_rate: Target sample rate
            listener_position: Spawn position for listener
            distance_gain: Apply 1/r distance attenuation
            progress_callback: Progress updates (source_idx, name, stage)

        Returns:
            True if successful, False otherwise
        '''
        try:
            logger.info(f"Creating ZYZ composition: {filename}")
            logger.info(f"Settings: Order={order}, Sample Rate={sample_rate}Hz")
            logger.info(f"Pre-mixing {len(sources)} sources into single ambisonic field")

            # Import encoding functions
            from core.shac.codec.encoders import encode_mono_source, convert_to_spherical

            # Calculate number of channels for this order
            num_channels = (order + 1) ** 2
            logger.debug(f"Ambisonic channels for order {order}: {num_channels}")

            # Initialize summed ambisonic field
            summed_field = None
            max_length = 0

            # Process each source
            for source_idx, source in enumerate(sources):
                source_id = source['id']
                export_name = source.get('export_name', source_id)

                if progress_callback:
                    progress_callback(source_idx, export_name, "Loading")

                if source_id not in self.loaded_samples:
                    logger.warning(f"Source {source_id} not found")
                    continue

                # Get audio data
                sample_info = self.loaded_samples[source_id]
                audio_data = sample_info['audio_data']
                original_sr = sample_info['sample_rate']

                # Resample if needed
                if original_sr != sample_rate:
                    if progress_callback:
                        progress_callback(source_idx, export_name, "Resampling")

                    logger.info(f"Resampling {export_name}: {original_sr}Hz -> {sample_rate}Hz")
                    if LIBROSA_AVAILABLE:
                        audio_data = librosa.resample(
                            audio_data,
                            orig_sr=original_sr,
                            target_sr=sample_rate
                        )
                    else:
                        logger.warning(f"Cannot resample {export_name} - librosa not available")

                # Convert to mono if needed
                if audio_data.ndim > 1:
                    if audio_data.shape[1] == 1:
                        audio_data = audio_data.flatten()
                    elif audio_data.shape[1] == 2:
                        # Mix stereo to mono
                        audio_data = audio_data[:, 0]
                    else:
                        audio_data = audio_data[:, 0]

                # Track max length for padding
                max_length = max(max_length, len(audio_data))

                if progress_callback:
                    progress_callback(source_idx, export_name, "Encoding")

                # Get position for this source
                position = source['position']
                logger.debug(f"Encoding {export_name} at position {position}")

                # Convert to spherical coordinates
                spherical_pos = convert_to_spherical(position)

                # Encode to ambisonics
                ambisonic_audio = encode_mono_source(
                    audio_data,
                    spherical_pos,
                    order,
                    apply_distance_gain=distance_gain
                )

                # Sum into the field (THE MAGIC OF ZYZ)
                if summed_field is None:
                    summed_field = ambisonic_audio.copy()
                    logger.debug(f"Initialized summed field with {export_name}: shape={summed_field.shape}")
                else:
                    # Pad shorter one if needed
                    # ambisonic_audio has shape (num_channels, n_samples)
                    current_len = summed_field.shape[1]
                    new_len = ambisonic_audio.shape[1]

                    if new_len > current_len:
                        # Pad summed_field to match new length
                        padding = np.zeros((num_channels, new_len - current_len))
                        summed_field = np.concatenate([summed_field, padding], axis=1)
                    elif new_len < current_len:
                        # Pad ambisonic_audio to match summed_field length
                        padding = np.zeros((num_channels, current_len - new_len))
                        ambisonic_audio = np.concatenate([ambisonic_audio, padding], axis=1)

                    # Add to summed field
                    summed_field += ambisonic_audio
                    logger.debug(f"Mixed {export_name} into field")

                if progress_callback:
                    progress_callback(source_idx, export_name, "Mixed")

            if summed_field is None:
                logger.error("No sources to export")
                return False

            if progress_callback:
                progress_callback(len(sources)-1, "All sources", "Finalizing")

            # Calculate final duration
            # summed_field has shape (num_channels, n_samples)
            duration = summed_field.shape[1] / sample_rate
            logger.info(f"Final mix duration: {duration:.2f}s ({summed_field.shape[1]} samples)")

            # Build source reference data for visualization
            source_references = []
            for source in sources:
                # Convert position to list (might be numpy array or tuple)
                pos = source['position']
                if hasattr(pos, 'tolist'):  # numpy array
                    pos = pos.tolist()
                else:
                    pos = list(pos)  # tuple or list

                source_references.append({
                    'name': source.get('export_name', source['id']),
                    'position': pos
                })

            # Create metadata
            # Convert listener_position to list (might be numpy array or tuple)
            listener_pos = listener_position
            if hasattr(listener_pos, 'tolist'):
                listener_pos = listener_pos.tolist()
            else:
                listener_pos = list(listener_pos) if listener_pos else [0, 0, 0]

            metadata = {
                'title': Path(filename).stem,
                'listener_spawn_position': listener_pos,
                'ambisonic_order': order,
                'sample_rate': sample_rate,
                'channels': num_channels,
                'duration': duration,
                'format': 'zyz',
                'format_version': '1.0',
                'created_by': 'SHAC Studio',
                'creation_time': time.strftime('%Y-%m-%d %H:%M:%S'),
                'distance_gain': distance_gain,
                'source_count': len(sources),
                'source_references': source_references  # For visualization in web player
            }

            # Save as ZYZ file
            output_path = self._save_zyz_file(
                summed_field,
                filename,
                metadata,
                sample_rate
            )

            file_size_mb = Path(output_path).stat().st_size / (1024 * 1024)
            logger.info(f"ZYZ composition saved: {output_path}")
            logger.info(f"File size: {file_size_mb:.1f} MB (~{len(sources)}x smaller than SHAC)")

            return True

        except Exception as e:
            logger.error(f"ZYZ export failed: {e}", exc_info=True)
            return False

    def _save_zyz_file(self, ambisonic_field, filename, metadata, sample_rate):
        '''
        Save pre-mixed ambisonic field as ZYZ file.

        Uses SHAC binary format with a single layer for web player compatibility.
        ZYZ files are just SHAC files with one pre-mixed layer.
        '''
        from core.shac.codec import SHACFileWriter
        from core.shac.codec.math_utils import AmbisonicNormalization

        # Ensure .zyz extension
        if not filename.endswith('.zyz'):
            filename += '.zyz'

        logger.debug(f"Writing ZYZ file: {filename}")
        logger.debug(f"Ambisonic field shape: {ambisonic_field.shape}")

        # The ambisonic_field from our encoding already has shape (n_channels, n_samples)
        # which is exactly what SHACFileWriter expects
        if ambisonic_field.ndim != 2:
            raise ValueError(f"Expected 2D ambisonic field, got shape {ambisonic_field.shape}")

        ambisonic_audio = ambisonic_field
        logger.debug(f"Using ambisonic field with shape: {ambisonic_audio.shape}")

        # Create SHAC file writer
        order = metadata.get('ambisonic_order', 3)
        writer = SHACFileWriter(
            order=order,
            sample_rate=sample_rate,
            normalization=AmbisonicNormalization.SN3D
        )

        # Add the single pre-mixed layer
        # Ensure all values are JSON-serializable (no numpy arrays)
        listener_spawn = metadata.get('listener_spawn_position', [0, 0, 0])
        if hasattr(listener_spawn, 'tolist'):
            listener_spawn = listener_spawn.tolist()

        layer_metadata = {
            'title': metadata.get('title', 'Untitled'),
            'format': 'zyz',
            'format_version': metadata.get('format_version', '1.0'),
            'created_by': metadata.get('created_by', 'SHAC Studio'),
            'listener_spawn_position': listener_spawn,
            'pre_mixed': True,
            'source_count': metadata.get('source_count', 0),
            'source_references': metadata.get('source_references', [])  # Critical for visualization!
        }

        writer.add_layer('premixed', ambisonic_audio, layer_metadata)

        # Write the file
        writer.write_file(filename, bit_depth=32)

        logger.debug(f"Wrote ZYZ file in SHAC format with single pre-mixed layer")
        return filename
    """
    # END OF LEGACY ZYZ CODE

    def cleanup(self):
        """Clean shutdown preserving any quality tracking data."""
        logger.info("Transparent Sampler Bridge shutdown - quality preserved")


# Create global instance for backward compatibility
class SamplerBridge(TransparentSamplerBridge):
    """Backward compatibility wrapper for transparent sampler with audio processing support."""
    
    def get_sources(self) -> List[str]:
        """Get list of loaded source names for audio processing panel."""
        return list(self.loaded_samples.keys())
    
    def get_source_audio(self, source_id: str) -> Optional[np.ndarray]:
        """Get full audio data for a source."""
        sample_info = self.loaded_samples.get(source_id)
        if sample_info:
            return sample_info.get('audio_data')
        return None
    
    def get_source_sample_rate(self, source_id: str) -> Optional[int]:
        """Get sample rate for a source."""
        sample_info = self.loaded_samples.get(source_id)
        if sample_info:
            return sample_info.get('sample_rate', 44100)
        return None
    
    def update_source_audio(self, source_id: str, audio_data: np.ndarray) -> bool:
        """Update audio data for a source after processing."""
        if source_id not in self.loaded_samples:
            return False
        
        try:
            # Get existing sample info
            sample_info = self.loaded_samples[source_id]
            old_data = sample_info.get('audio_data')
            
            # Update the audio data while preserving metadata
            sample_info['audio_data'] = audio_data
            sample_info['duration'] = len(audio_data) / sample_info['sample_rate']
            sample_info['frames'] = len(audio_data)
            sample_info['modifications'].append(f"Audio processed at {time.time()}")
            
            # Emit state change event
            state_manager.emit(
                StateEvent.AUDIO_DATA_EDITED,
                source_id=source_id,
                data={
                    'old_data': old_data,
                    'new_data': audio_data
                },
                component='TransparentSamplerBridge'
            )
            
            # Update main window if available
            if self.main_window:
                self.main_window.notify_source_update(source_id)

            logger.info(f"Updated audio for {source_id}")
            return True

        except Exception as e:
            logger.error(f"Failed to update audio: {e}")
            return False

    def on_audio_data_edited(self, change):
        """Handle audio data edited event for undo/redo support."""
        # Only respond to events from UndoRedo system to avoid infinite loops
        if change.component == 'UndoRedo' and 'new_data' in change.data:
            source_id = change.data.get('source')
            new_data = change.data['new_data']

            if source_id and source_id in self.loaded_samples and new_data is not None:
                # Don't emit another event - just update the data directly
                sample_info = self.loaded_samples[source_id]
                sample_info['audio_data'] = new_data
                sample_info['duration'] = len(new_data) / sample_info['sample_rate']
                sample_info['frames'] = len(new_data)
                sample_info['modifications'].append(f"Undo/Redo at {time.time()}")

                # Update main window if available
                if self.main_window:
                    self.main_window.notify_source_update(source_id)

                logger.info(f"Restored audio data for {source_id} via undo/redo")

    def rename_source(self, source_id: str, new_name: str) -> bool:
        """Rename a loaded source."""
        if source_id not in self.loaded_samples:
            logger.error(f"Source {source_id} not found for renaming")
            return False

        try:
            # Update the sample info to include the new name
            sample_info = self.loaded_samples[source_id]
            old_name = sample_info.get('name', source_id)

            # Store new name in sample info (consistent with add_audio_to_project)
            sample_info['name'] = new_name
            sample_info['modifications'].append(f"Renamed from '{old_name}' to '{new_name}' at {time.time()}")

            # Emit state change event
            self.emit_state_change(StateEvent.SOURCE_RENAMED, source_id, {
                'old_name': old_name,
                'new_name': new_name
            })

            logger.info(f"Renamed source {source_id}: '{old_name}' → '{new_name}'")
            return True

        except Exception as e:
            logger.error(f"Failed to rename source: {e}")
            return False
    
    def get_source_info(self, source_id: str) -> Optional[Dict]:
        """Get information about a source including position, volume, etc."""
        if source_id not in self.loaded_samples:
            return None
            
        sample_info = self.loaded_samples[source_id]
        
        # Return info in format expected by Source Control Panel
        return {
            'id': source_id,
            'name': sample_info.get('name', 'Unknown'),
            'position': sample_info.get('position', {'x': 0, 'y': 0, 'z': 0}),
            'volume': sample_info.get('volume', 1.0),
            'gain': sample_info.get('gain', 1.0),
            'pan': sample_info.get('pan', 0.0),
            'muted': sample_info.get('muted', False),
            'soloed': sample_info.get('soloed', False)
        }
    
    def set_source_mute(self, source_id: str, muted: bool) -> bool:
        """Set mute state for a source."""
        if source_id not in self.loaded_samples:
            return False
            
        self.loaded_samples[source_id]['muted'] = muted
        
        # Update audio engine if available
        if self.main_window and hasattr(self.main_window, 'audio_engine'):
            if muted:
                self.main_window.audio_engine.mute_source(source_id)
            else:
                self.main_window.audio_engine.unmute_source(source_id)
                
        return True
    
    def set_source_solo(self, source_id: str, soloed: bool) -> bool:
        """Set solo state for a source."""
        if source_id not in self.loaded_samples:
            return False
            
        self.loaded_samples[source_id]['soloed'] = soloed
        
        # Update audio engine if available
        if self.main_window and hasattr(self.main_window, 'audio_engine'):
            if soloed:
                self.main_window.audio_engine.solo_source(source_id)
            else:
                # Check if any sources are still soloed
                any_soloed = any(s.get('soloed', False) for s in self.loaded_samples.values())
                if not any_soloed:
                    self.main_window.audio_engine.unsolo_all_sources()
                    
        return True
    
    def set_source_volume(self, source_id: str, volume: float) -> bool:
        """Set volume for a source (0.0 to 1.0)."""
        if source_id not in self.loaded_samples:
            return False
            
        self.loaded_samples[source_id]['volume'] = volume
        
        # Update audio engine if available
        if self.main_window and hasattr(self.main_window, 'audio_engine'):
            self.main_window.audio_engine.set_source_volume(source_id, volume)
            
        return True
    
    def set_source_gain(self, source_id: str, gain: float) -> bool:
        """Set gain for a source (linear multiplier)."""
        if source_id not in self.loaded_samples:
            return False
            
        self.loaded_samples[source_id]['gain'] = gain
        
        # Update audio engine if available
        if self.main_window and hasattr(self.main_window, 'audio_engine'):
            self.main_window.audio_engine.set_source_gain(source_id, gain)
            
        return True
    
    def set_source_pan(self, source_id: str, pan: float) -> bool:
        """Set pan for a source (-1.0 to 1.0)."""
        if source_id not in self.loaded_samples:
            return False
            
        self.loaded_samples[source_id]['pan'] = pan
        
        # Update audio engine if available
        if self.main_window and hasattr(self.main_window, 'audio_engine'):
            self.main_window.audio_engine.set_source_pan(source_id, pan)
            
        return True
    
    def set_source_position(self, source_id: str, x: float, y: float, z: float) -> bool:
        """Set 3D position for a source."""
        if source_id not in self.loaded_samples:
            return False
            
        self.loaded_samples[source_id]['position'] = {'x': x, 'y': y, 'z': z}
        
        # Update audio engine if available
        if self.main_window and hasattr(self.main_window, 'audio_engine'):
            self.main_window.audio_engine.set_source_position(source_id, x, y, z)
            
        # Update spatial view if available
        if hasattr(self, 'spatial_view') and self.spatial_view:
            self.spatial_view.update_source_position(source_id, x, y, z)
            
        return True