"""
SHAC Studio - Professional Dark Theme
Inspired by Pro Tools, Ableton Live, and Logic Pro aesthetics
"""

# Base Colors
BACKGROUND_PRIMARY = '#1E1E1E'      # Main background
BACKGROUND_SECONDARY = '#2D2D2D'    # Panels/cards
BACKGROUND_TERTIARY = '#3A3A3A'     # Hover states
BACKGROUND_INPUT = '#252525'        # Input fields
BACKGROUND_LISTBOX = '#3C3C3C'      # Softer mid-gray for listboxes

# Accent Colors (Spatial audio theme - blue/cyan for "space")
ACCENT_PRIMARY = '#4A9EFF'          # Primary actions, selected items
ACCENT_HOVER = '#6BB1FF'            # Hover states
ACCENT_PRESSED = '#3A8EEF'          # Pressed states
ACCENT_GLOW = '#5AA5FF'             # Subtle glows

# Text Colors
TEXT_PRIMARY = '#E0E0E0'            # Main text - white/light gray for readability
TEXT_SECONDARY = '#B0B0B0'          # Secondary text - medium gray
TEXT_DISABLED = '#707070'           # Disabled text
TEXT_HINT = '#909090'               # Hints/placeholders
TEXT_ACCENT = '#3498DB'             # Beautiful blue for headers and important text!

# Status Colors
SUCCESS = '#4CAF50'                 # Green for active/playing
WARNING = '#FFC107'                 # Yellow for warnings
ERROR = '#F44336'                   # Red for errors
MUTED = '#757575'                   # Gray for muted sources
RECORDING = '#FF5252'               # Recording state

# UI Elements
BORDER = '#404040'                  # Panel borders
DIVIDER = '#333333'                 # Separators
SLIDER_TRACK = '#505050'            # Slider backgrounds
SLIDER_FILL = ACCENT_PRIMARY        # Slider active portion
BUTTON_NORMAL = '#3C3C3C'           # Button background
BUTTON_HOVER = '#4A4A4A'            # Button hover
BUTTON_PRESSED = '#2A2A2A'          # Button pressed

# Special Elements
WAVEFORM = ACCENT_PRIMARY           # Waveform display color
SPATIAL_GRID = '#2A2A2A'            # 3D grid lines
SPATIAL_AXIS = '#505050'            # Axis lines
SOURCE_MARKER = ACCENT_PRIMARY      # Audio source markers
LISTENER_MARKER = SUCCESS           # Listener position marker


class Theme:
    """Professional dark theme for SHAC Studio"""

    # TTK Style configurations for tkinter
    TTK_STYLE = {
        'TFrame': {
            'configure': {
                'background': BACKGROUND_SECONDARY,
                'borderwidth': 0,
                'relief': 'flat'
            }
        },
        'TLabel': {
            'configure': {
                'background': BACKGROUND_SECONDARY,
                'foreground': TEXT_PRIMARY,
                'borderwidth': 0,
                'relief': 'flat'
            }
        },
        'TLabelframe': {
            'configure': {
                'background': BACKGROUND_SECONDARY,
                'foreground': TEXT_ACCENT,
                'borderwidth': 1,
                'relief': 'flat',
                'labeloutside': False
            }
        },
        'TLabelframe.Label': {
            'configure': {
                'background': BACKGROUND_SECONDARY,
                'foreground': TEXT_ACCENT
            }
        },
        'TButton': {
            'configure': {
                'background': BUTTON_NORMAL,
                'foreground': TEXT_PRIMARY,
                'borderwidth': 1,
                'relief': 'flat'
            },
            'map': {
                'background': [
                    ('active', BUTTON_HOVER),
                    ('pressed', BUTTON_PRESSED),
                    ('disabled', BACKGROUND_TERTIARY)
                ],
                'foreground': [('disabled', TEXT_DISABLED)]
            }
        },
        'Accent.TButton': {
            'configure': {
                'background': ACCENT_PRIMARY,
                'foreground': TEXT_PRIMARY,
                'borderwidth': 0,
                'relief': 'flat'
            },
            'map': {
                'background': [
                    ('active', ACCENT_HOVER),
                    ('pressed', ACCENT_PRESSED)
                ]
            }
        },
        'TEntry': {
            'configure': {
                'fieldbackground': BACKGROUND_INPUT,
                'foreground': TEXT_PRIMARY,
                'borderwidth': 1,
                'relief': 'flat'
            }
        },
        'TScale': {
            'configure': {
                'background': BACKGROUND_SECONDARY,
                'troughcolor': SLIDER_TRACK,
                'borderwidth': 0,
                'relief': 'flat'
            }
        }
    }

    @staticmethod
    def apply_to_widget(widget, bg=BACKGROUND_SECONDARY, fg=TEXT_PRIMARY):
        """Apply theme colors to a tkinter widget"""
        try:
            widget.config(bg=bg, fg=fg)
        except:
            pass  # Some widgets don't support all options

    @staticmethod
    def configure_ttk_style(style):
        """Configure a ttk.Style object with the professional theme"""
        # Set the theme to 'clam' or 'alt' which allows better customization
        try:
            style.theme_use('clam')
        except:
            pass

        for style_name, config in Theme.TTK_STYLE.items():
            if 'configure' in config:
                style.configure(style_name, **config['configure'])
            if 'map' in config:
                style.map(style_name, **config['map'])

        # Kill those annoying striped backgrounds!
        style.layout('TFrame', [])
        style.layout('TLabel', [])

        # Force consistent backgrounds everywhere
        style.configure('.', background=BACKGROUND_SECONDARY)
        style.map('TFrame', background=[('!disabled', BACKGROUND_SECONDARY)])
        style.map('TLabel', background=[('!disabled', BACKGROUND_SECONDARY)])


# Font configurations
FONT_FAMILY = 'Segoe UI' if 'win' in str(__import__('sys').platform) else 'SF Pro Display' if 'darwin' in str(__import__('sys').platform) else 'Ubuntu'
FONT_SIZE_SMALL = 9
FONT_SIZE_NORMAL = 10
FONT_SIZE_LARGE = 12
FONT_SIZE_TITLE = 14

FONT_NORMAL = (FONT_FAMILY, FONT_SIZE_NORMAL)
FONT_BOLD = (FONT_FAMILY, FONT_SIZE_NORMAL, 'bold')
FONT_SMALL = (FONT_FAMILY, FONT_SIZE_SMALL)
FONT_LARGE = (FONT_FAMILY, FONT_SIZE_LARGE)
FONT_TITLE = (FONT_FAMILY, FONT_SIZE_TITLE, 'bold')
