"""
Audio Processing Panel - Simplified audio modifications for SHAC Studio

This panel provides essential audio processing tools without waveform editing:
- Sound Normalization (Peak and RMS)
- Volume Normalization
- Gain adjustment
- Trim to length
- Add silence at specific timecodes
"""

import logging
import tkinter as tk
from tkinter import ttk, messagebox
import numpy as np
from pathlib import Path

# Set up logger
logger = logging.getLogger(__name__)
import sys

from ..dockable_panel import DockablePanel
from .. import theme
from ..icons import get_icon
from ..theme_helpers import create_themed_button, create_themed_label, create_themed_frame

# Import state management
sys.path.insert(0, str(Path(__file__).parent.parent.parent))
from core.state_manager import StatefulComponent, StateEvent, state_manager, state_aware_method


class AudioProcessingPanel(DockablePanel, StatefulComponent):
    """Simplified audio processing panel focused on essential modifications."""
    
    def __init__(self, audio_bridge, parent_window=None):
        """Initialize the audio processing panel."""
        DockablePanel.__init__(self, "audio_processing", "Audio Processing", parent_window)
        StatefulComponent.__init__(self, "AudioProcessingPanel")
        
        self.audio_bridge = audio_bridge
        self.selected_source = None
        self.sources = {}  # Track loaded sources
        
        # Single source editing - no mute/solo needed
        
        # UI elements (will be created in setup_content)
        self.source_listbox = None
        self.selected_label = None
        
        # Source preview elements
        self.preview_canvas = None
        self.preview_info_label = None
        self.is_playing = False
        self.current_position = 0.0
        self.audio_duration = 0.0
        self.playback_line = None
        self.preview_audio_data = None
        self.preview_sample_rate = None
        
        # Default geometry (increased height for preview section)
        self.default_width = 450
        self.default_height = 650
        self.default_x = 400
        self.default_y = 300
        
        # Audio engine reference (will be set by main window)
        self.audio_engine = None
        
        # Subscribe to state events
        self._setup_state_subscriptions()
        
    def _setup_state_subscriptions(self):
        """Set up state event subscriptions."""
        self.subscribe_to_state(StateEvent.SOURCE_ADDED, 'on_source_added')
        self.subscribe_to_state(StateEvent.SOURCE_REMOVED, 'on_source_removed')
        self.subscribe_to_state(StateEvent.SOURCE_RENAMED, 'on_source_renamed')
        self.subscribe_to_state(StateEvent.SELECTION_CHANGED, 'on_selection_changed')
        
    def setup_content(self):
        """Set up the panel content with professional theme."""
        # Apply theme
        self.content_frame.config(bg=theme.BACKGROUND_SECONDARY)

        # Main container with padding
        main_frame = create_themed_frame(self.content_frame)
        main_frame.pack(fill='both', expand=True, padx=10, pady=10)
        
        # Title and description
        title_label = ttk.Label(
            main_frame,
            text="Audio Processing Tools",
            font=('Arial', 12, 'bold')
        )
        title_label.pack(pady=(0, 5))
        
        desc_label = ttk.Label(
            main_frame,
            text="Process audio sources before spatial positioning",
            font=('Arial', 9)
        )
        desc_label.pack(pady=(0, 15))
        
        # Create two-column layout
        columns_frame = ttk.Frame(main_frame)
        columns_frame.pack(fill='both', expand=True)
        
        # Left column - Source list
        left_frame = ttk.Frame(columns_frame)
        left_frame.pack(side='left', fill='both', expand=True, padx=(0, 5))
        
        # Source header with refresh button
        source_header = ttk.Frame(left_frame)
        source_header.pack(fill='x')
        
        source_label = ttk.Label(source_header, text="Audio Sources", font=('Arial', 10, 'bold'))
        source_label.pack(side='left')
        
        # Themed refresh button
        refresh_icon = get_icon('settings', size=16, color=theme.TEXT_PRIMARY)
        self.refresh_button = tk.Button(
            source_header,
            text=" Refresh",
            image=refresh_icon,
            compound='left',
            command=self.manual_refresh_sources,
            bg=theme.BUTTON_NORMAL,
            fg=theme.TEXT_PRIMARY,
            activebackground=theme.BUTTON_HOVER,
            relief='flat',
            borderwidth=0,
            padx=8,
            pady=4,
            font=theme.FONT_SMALL,
            cursor='hand2'
        )
        self.refresh_button.image = refresh_icon  # Keep reference
        self.refresh_button.pack(side='right')

        # Create preview control icons (white for visibility on colored buttons)
        self.play_icon = get_icon('play', size=16, color='#FFFFFF')
        self.pause_icon = get_icon('pause', size=16, color='#FFFFFF')
        self.stop_icon = get_icon('stop', size=16, color='#FFFFFF')
        
        
        # Source listbox with scrollbar
        list_frame = ttk.Frame(left_frame)
        list_frame.pack(fill='both', expand=True, pady=(5, 0))
        
        scrollbar = ttk.Scrollbar(list_frame)
        scrollbar.pack(side='right', fill='y')
        
        # Themed source listbox with softer background
        self.source_listbox = tk.Listbox(
            list_frame,
            selectmode='single',
            yscrollcommand=scrollbar.set,
            font=theme.FONT_NORMAL,
            height=10,
            bg=theme.BACKGROUND_LISTBOX,  # Softer mid-gray
            fg=theme.TEXT_PRIMARY,
            selectbackground=theme.ACCENT_PRIMARY,
            selectforeground='#FFFFFF',
            borderwidth=0,
            highlightthickness=0
        )
        self.source_listbox.pack(side='left', fill='both', expand=True)
        scrollbar.config(command=self.source_listbox.yview)
        
        # Bind selection event
        self.source_listbox.bind('<<ListboxSelect>>', self.on_source_selected)

        # Create right-click context menu
        self._create_context_menu()

        # Bind right-click to show context menu
        self.source_listbox.bind('<Button-3>', self.show_context_menu)  # Right-click
        self.source_listbox.bind('<Button-2>', self.show_context_menu)  # macOS

        # Right column - Processing options
        right_frame = ttk.Frame(columns_frame)
        right_frame.pack(side='right', fill='both', expand=True, padx=(5, 0))
        
        options_label = ttk.Label(right_frame, text="Processing Options", font=('Arial', 10, 'bold'))
        options_label.pack()
        
        # Selected source info
        self.selected_label = ttk.Label(right_frame, text="No source selected", font=('Arial', 9, 'italic'))
        self.selected_label.pack(pady=(5, 10))
        
        # Removed mute/solo controls - not needed for single source editing
        
        # Processing sections
        self.create_normalization_section(right_frame)
        self.create_gain_section(right_frame)
        self.create_trim_section(right_frame)
        self.create_silence_section(right_frame)
        
        # Source preview section (spans both columns)
        self.create_source_preview_section(main_frame)
        
        # Load current sources
        self.refresh_sources()

    def _create_context_menu(self):
        """Create right-click context menu for source list."""
        self.context_menu = tk.Menu(
            self.source_listbox,
            tearoff=0,
            bg=theme.BACKGROUND_SECONDARY,
            fg=theme.TEXT_PRIMARY,
            activebackground=theme.ACCENT_PRIMARY,
            activeforeground='#FFFFFF',
            borderwidth=0
        )

        # Add menu items - just refresh for this panel
        self.context_menu.add_command(
            label="Refresh Sources",
            command=self.manual_refresh_sources
        )

    def show_context_menu(self, event):
        """Show context menu at cursor position."""
        # Select the item under cursor
        index = self.source_listbox.nearest(event.y)
        if index >= 0:
            self.source_listbox.selection_clear(0, tk.END)
            self.source_listbox.selection_set(index)
            # Trigger selection event
            self.on_source_selected(None)
            # Show menu at cursor position
            try:
                self.context_menu.tk_popup(event.x_root, event.y_root)
            finally:
                self.context_menu.grab_release()

    # Playback controls removed - single source editing doesn't need mute/solo/volume

    def create_normalization_section(self, parent):
        """Create normalization controls section."""
        frame = ttk.LabelFrame(parent, text="Normalization", padding="10")
        frame.pack(fill='x', pady=(0, 10))
        
        # Peak normalization
        peak_frame = ttk.Frame(frame)
        peak_frame.pack(fill='x', pady=(0, 5))
        
        ttk.Label(peak_frame, text="Peak Level (dB):").pack(side='left')
        self.peak_level_var = tk.StringVar(value="-3.0")
        peak_entry = ttk.Entry(peak_frame, textvariable=self.peak_level_var, width=8)
        peak_entry.pack(side='left', padx=5)
        ttk.Button(peak_frame, text="Normalize Peak", command=self.normalize_peak).pack(side='left', padx=5)
        
        # RMS normalization
        rms_frame = ttk.Frame(frame)
        rms_frame.pack(fill='x')
        
        ttk.Label(rms_frame, text="RMS Level (dB):").pack(side='left')
        self.rms_level_var = tk.StringVar(value="-18.0")
        rms_entry = ttk.Entry(rms_frame, textvariable=self.rms_level_var, width=8)
        rms_entry.pack(side='left', padx=5)
        ttk.Button(rms_frame, text="Normalize RMS", command=self.normalize_rms).pack(side='left', padx=5)
        
    def create_gain_section(self, parent):
        """Create gain adjustment section."""
        frame = ttk.LabelFrame(parent, text="Gain Adjustment", padding="10")
        frame.pack(fill='x', pady=(0, 10))
        
        gain_frame = ttk.Frame(frame)
        gain_frame.pack(fill='x')
        
        ttk.Label(gain_frame, text="Gain (dB):").pack(side='left')
        self.gain_var = tk.StringVar(value="0.0")
        gain_entry = ttk.Entry(gain_frame, textvariable=self.gain_var, width=8)
        gain_entry.pack(side='left', padx=5)
        ttk.Button(gain_frame, text="Apply Gain", command=self.apply_gain).pack(side='left', padx=5)
        
        # Quick gain buttons
        quick_frame = ttk.Frame(frame)
        quick_frame.pack(fill='x', pady=(5, 0))
        ttk.Label(quick_frame, text="Quick:").pack(side='left')
        for db in ["-6", "-3", "+3", "+6"]:
            ttk.Button(
                quick_frame, 
                text=f"{db}dB",
                command=lambda d=db: self.quick_gain(d),
                width=6
            ).pack(side='left', padx=2)
            
    def create_trim_section(self, parent):
        """Create trim controls section."""
        frame = ttk.LabelFrame(parent, text="Trim Audio", padding="10")
        frame.pack(fill='x', pady=(0, 10))
        
        # Trim to length
        length_frame = ttk.Frame(frame)
        length_frame.pack(fill='x', pady=(0, 5))
        
        ttk.Label(length_frame, text="Length (seconds):").pack(side='left')
        self.trim_length_var = tk.StringVar(value="30.0")
        length_entry = ttk.Entry(length_frame, textvariable=self.trim_length_var, width=8)
        length_entry.pack(side='left', padx=5)
        ttk.Button(length_frame, text="Trim to Length", command=self.trim_to_length).pack(side='left', padx=5)
        
        # Trim silence
        ttk.Button(frame, text="Trim Start/End Silence", command=self.trim_silence).pack(pady=(5, 0))
        
    def create_silence_section(self, parent):
        """Create silence insertion section."""
        frame = ttk.LabelFrame(parent, text="Add Silence", padding="10")
        frame.pack(fill='x')
        
        # Add silence at timecode
        time_frame = ttk.Frame(frame)
        time_frame.pack(fill='x', pady=(0, 5))
        
        ttk.Label(time_frame, text="Position (seconds):").pack(side='left')
        self.silence_position_var = tk.StringVar(value="0.0")
        pos_entry = ttk.Entry(time_frame, textvariable=self.silence_position_var, width=8)
        pos_entry.pack(side='left', padx=5)
        
        duration_frame = ttk.Frame(frame)
        duration_frame.pack(fill='x')
        
        ttk.Label(duration_frame, text="Duration (seconds):").pack(side='left')
        self.silence_duration_var = tk.StringVar(value="1.0")
        dur_entry = ttk.Entry(duration_frame, textvariable=self.silence_duration_var, width=8)
        dur_entry.pack(side='left', padx=5)
        
        ttk.Button(frame, text="Add Silence", command=self.add_silence).pack(pady=(5, 0))
        
    def create_source_preview_section(self, parent):
        """Create source preview section with timeline and playback controls."""
        # Separator
        separator = ttk.Separator(parent, orient='horizontal')
        separator.pack(fill='x', pady=(15, 10))
        
        # Preview section frame
        preview_frame = ttk.LabelFrame(parent, text="Source Preview", padding="10")
        preview_frame.pack(fill='x', pady=(0, 10))
        
        # Source info display
        self.preview_info_label = ttk.Label(
            preview_frame, 
            text="No source selected for preview",
            font=('Arial', 9, 'italic')
        )
        self.preview_info_label.pack(pady=(0, 10))
        
        # Playback controls
        controls_frame = tk.Frame(preview_frame, bg=theme.BACKGROUND_SECONDARY)
        controls_frame.pack(fill='x', pady=(0, 10))

        # Control buttons with icons
        self.play_button = tk.Button(
            controls_frame,
            text=" Play",
            image=self.play_icon,
            compound='left',
            command=self.preview_play,
            bg=theme.BUTTON_NORMAL,
            fg=theme.TEXT_PRIMARY,
            activebackground=theme.BUTTON_HOVER,
            activeforeground=theme.TEXT_PRIMARY,
            relief='flat',
            borderwidth=0,
            padx=10,
            pady=6,
            font=theme.FONT_NORMAL,
            cursor='hand2'
        )
        self.play_button.pack(side='left', padx=(0, 5))

        self.pause_button = tk.Button(
            controls_frame,
            text=" Pause",
            image=self.pause_icon,
            compound='left',
            command=self.preview_pause,
            bg=theme.BUTTON_NORMAL,
            fg=theme.TEXT_PRIMARY,
            activebackground=theme.BUTTON_HOVER,
            activeforeground=theme.TEXT_PRIMARY,
            relief='flat',
            borderwidth=0,
            padx=10,
            pady=6,
            font=theme.FONT_NORMAL,
            cursor='hand2'
        )
        self.pause_button.pack(side='left', padx=(0, 5))

        self.stop_button = tk.Button(
            controls_frame,
            text=" Stop",
            image=self.stop_icon,
            compound='left',
            command=self.preview_stop,
            bg=theme.BUTTON_NORMAL,
            fg=theme.TEXT_PRIMARY,
            activebackground=theme.BUTTON_HOVER,
            activeforeground=theme.TEXT_PRIMARY,
            relief='flat',
            borderwidth=0,
            padx=10,
            pady=6,
            font=theme.FONT_NORMAL,
            cursor='hand2'
        )
        self.stop_button.pack(side='left', padx=(0, 5))

        self.restart_button = tk.Button(
            controls_frame,
            text=" Restart",
            command=self.preview_restart,
            bg=theme.BUTTON_NORMAL,
            fg=theme.TEXT_PRIMARY,
            activebackground=theme.BUTTON_HOVER,
            activeforeground=theme.TEXT_PRIMARY,
            relief='flat',
            borderwidth=0,
            padx=10,
            pady=6,
            font=theme.FONT_NORMAL,
            cursor='hand2'
        )
        self.restart_button.pack(side='left', padx=(0, 10))
        
        # Time display
        self.time_label = ttk.Label(controls_frame, text="0.00s / 0.00s", font=('Arial', 9))
        self.time_label.pack(side='right')
        
        # Timeline frame
        timeline_frame = ttk.Frame(preview_frame)
        timeline_frame.pack(fill='x')
        
        # Beautiful themed timeline canvas
        self.preview_canvas = tk.Canvas(
            timeline_frame,
            height=50,
            bg=theme.BACKGROUND_SECONDARY,
            highlightthickness=0,
            bd=0
        )
        self.preview_canvas.pack(fill='x', pady=(5, 0))
        
        # Bind canvas events for seeking
        self.preview_canvas.bind('<Button-1>', self.on_timeline_click)
        self.preview_canvas.bind('<Configure>', self.on_canvas_resize)
        
        # Initially disable controls
        self.update_preview_controls()
        
    def on_source_selected(self, event):
        """Handle source selection from listbox."""
        selection = self.source_listbox.curselection()
        if selection:
            index = selection[0]
            display_name = self.source_listbox.get(index)
            # Get the actual source ID from our mapping
            source_id = self.sources.get(display_name, display_name)
            
            self.selected_source = source_id  # Store the actual source ID
            self.selected_label.config(text=f"Selected: {display_name}")
            
            # Emit selection event
            if hasattr(state_manager, 'emit_event'):
                state_manager.emit_event(StateEvent.SELECTION_CHANGED, {
                    'component': self.component_name,
                    'source': source_id,
                    'source_id': source_id
                })
            
            # Load audio for preview using the actual source ID
            self.load_preview_audio(source_id)
            
            # Single source editing - no additional controls needed
        else:
            self.selected_source = None
            self.selected_label.config(text="No source selected")
            self.clear_preview_audio()
            
            # Single source editing - no additional controls to disable
            
    def refresh_sources(self):
        """Refresh the source list from audio bridge."""
        # Check if UI is set up yet
        if not self.source_listbox:
            return
            
        self.source_listbox.delete(0, tk.END)
        self.sources.clear()  # Clear the name-to-ID mapping
        
        if self.audio_bridge:
            sources = self.audio_bridge.get_sources()
            for source_id in sources:
                # Get the actual source name instead of UUID
                source_info = self.audio_bridge.get_source_info(source_id)
                if source_info:
                    display_name = source_info.get('name', source_id)
                    self.source_listbox.insert(tk.END, display_name)
                    # Store mapping of display name to source ID
                    self.sources[display_name] = source_id
                else:
                    self.source_listbox.insert(tk.END, source_id)
                
    def manual_refresh_sources(self):
        """Manually refresh sources while preserving current work and selection."""
        if not self.source_listbox:
            return
            
        # Save current selection
        current_selection = self.selected_source
        
        # Clear and reload source list
        self.source_listbox.delete(0, tk.END)
        self.sources.clear()  # Clear the name-to-ID mapping
        
        if self.audio_bridge:
            sources = self.audio_bridge.get_sources()
            for source_id in sources:
                # Get the actual source name instead of UUID
                source_info = self.audio_bridge.get_source_info(source_id)
                if source_info:
                    display_name = source_info.get('name', source_id)
                    self.source_listbox.insert(tk.END, display_name)
                    # Store mapping of display name to source ID
                    self.sources[display_name] = source_id
                else:
                    self.source_listbox.insert(tk.END, source_id)
                
        # Try to restore previous selection if it still exists (current_selection is a source ID)
        if current_selection:
            # Find the display name that maps to this source ID
            for display_name, source_id in self.sources.items():
                if source_id == current_selection:
                    # Find this display name in the listbox
                    for i in range(self.source_listbox.size()):
                        if self.source_listbox.get(i) == display_name:
                            self.source_listbox.selection_clear(0, tk.END)
                            self.source_listbox.selection_set(i)
                            self.source_listbox.see(i)
                            # Keep the same selected_source and preview
                            break
                    break
            else:
                # Previous selection no longer exists, clear selection
                self.selected_source = None
                self.selected_label.config(text="No source selected")
                self.clear_preview_audio()
                
        logger.debug(f"Refreshed source list - found {self.source_listbox.size()} sources")
                
    # Audio processing methods
    
    def normalize_peak(self):
        """Normalize audio to target peak level."""
        if not self.selected_source:
            messagebox.showwarning("No Source", "Please select an audio source first")
            return
            
        try:
            target_db = float(self.peak_level_var.get())
            target_linear = 10 ** (target_db / 20.0)
            
            # Get audio data
            audio_data = self.audio_bridge.get_source_audio(self.selected_source)
            if audio_data is None:
                return
                
            # Find peak
            peak = np.max(np.abs(audio_data))
            if peak > 0:
                # Calculate gain needed
                gain = target_linear / peak
                # Apply gain
                normalized = audio_data * gain
                # Update audio
                self.audio_bridge.update_source_audio(self.selected_source, normalized)
                
                messagebox.showinfo("Success", f"Normalized {self.selected_source} to {target_db} dB peak")
                
        except ValueError:
            messagebox.showerror("Invalid Value", "Please enter a valid number for peak level")
        except Exception as e:
            messagebox.showerror("Error", f"Error normalizing: {str(e)}")
            
    def normalize_rms(self):
        """Normalize audio to target RMS level."""
        if not self.selected_source:
            messagebox.showwarning("No Source", "Please select an audio source first")
            return
            
        try:
            target_db = float(self.rms_level_var.get())
            target_linear = 10 ** (target_db / 20.0)
            
            # Get audio data
            audio_data = self.audio_bridge.get_source_audio(self.selected_source)
            if audio_data is None:
                return
                
            # Calculate RMS
            rms = np.sqrt(np.mean(audio_data ** 2))
            if rms > 0:
                # Calculate gain needed
                gain = target_linear / rms
                # Apply gain
                normalized = audio_data * gain
                # Prevent clipping
                peak = np.max(np.abs(normalized))
                if peak > 1.0:
                    normalized = normalized / peak
                    
                # Update audio
                self.audio_bridge.update_source_audio(self.selected_source, normalized)
                
                messagebox.showinfo("Success", f"Normalized {self.selected_source} to {target_db} dB RMS")
                
        except ValueError:
            messagebox.showerror("Invalid Value", "Please enter a valid number for RMS level")
        except Exception as e:
            messagebox.showerror("Error", f"Error normalizing: {str(e)}")
            
    def apply_gain(self):
        """Apply gain adjustment to selected source."""
        if not self.selected_source:
            messagebox.showwarning("No Source", "Please select an audio source first")
            return
            
        try:
            gain_db = float(self.gain_var.get())
            gain_linear = 10 ** (gain_db / 20.0)
            
            # Get audio data
            audio_data = self.audio_bridge.get_source_audio(self.selected_source)
            if audio_data is None:
                return
                
            # Apply gain
            gained = audio_data * gain_linear
            
            # Prevent clipping
            peak = np.max(np.abs(gained))
            if peak > 1.0:
                gained = gained / peak
                messagebox.showinfo("Clipping Prevention", f"Audio was normalized to prevent clipping")
                
            # Update audio
            self.audio_bridge.update_source_audio(self.selected_source, gained)
            
            messagebox.showinfo("Success", f"Applied {gain_db} dB gain to {self.selected_source}")
            
        except ValueError:
            messagebox.showerror("Invalid Value", "Please enter a valid number for gain")
        except Exception as e:
            messagebox.showerror("Error", f"Error applying gain: {str(e)}")
            
    def quick_gain(self, db_str):
        """Apply quick gain adjustment."""
        self.gain_var.set(db_str)
        self.apply_gain()
        
    def trim_to_length(self):
        """Trim audio to specified length."""
        if not self.selected_source:
            messagebox.showwarning("No Source", "Please select an audio source first")
            return
            
        try:
            target_length = float(self.trim_length_var.get())
            
            # Get audio data and sample rate
            audio_data = self.audio_bridge.get_source_audio(self.selected_source)
            sample_rate = self.audio_bridge.get_source_sample_rate(self.selected_source)
            
            if audio_data is None or sample_rate is None:
                return
                
            # Calculate target samples
            target_samples = int(target_length * sample_rate)
            current_samples = len(audio_data)
            
            if target_samples >= current_samples:
                messagebox.showinfo("No Change", f"Audio is already shorter than {target_length} seconds")
                return
                
            # Trim audio
            trimmed = audio_data[:target_samples]
            
            # Update audio
            self.audio_bridge.update_source_audio(self.selected_source, trimmed)
            
            messagebox.showinfo("Success", f"Trimmed {self.selected_source} to {target_length} seconds")
            
        except ValueError:
            messagebox.showerror("Invalid Value", "Please enter a valid number for length")
        except Exception as e:
            messagebox.showerror("Error", f"Error trimming: {str(e)}")
            
    def trim_silence(self):
        """Trim silence from start and end of audio."""
        if not self.selected_source:
            messagebox.showwarning("No Source", "Please select an audio source first")
            return
            
        try:
            # Get audio data
            audio_data = self.audio_bridge.get_source_audio(self.selected_source)
            if audio_data is None:
                return
                
            # Define silence threshold (adjust as needed)
            threshold = 0.001  # -60 dB
            
            # Find first non-silent sample
            start_idx = 0
            for i, sample in enumerate(audio_data):
                if abs(sample) > threshold:
                    start_idx = i
                    break
                    
            # Find last non-silent sample
            end_idx = len(audio_data) - 1
            for i in range(len(audio_data) - 1, -1, -1):
                if abs(audio_data[i]) > threshold:
                    end_idx = i
                    break
                    
            # Trim audio
            if start_idx > 0 or end_idx < len(audio_data) - 1:
                trimmed = audio_data[start_idx:end_idx + 1]
                self.audio_bridge.update_source_audio(self.selected_source, trimmed)
                
                # Calculate trimmed duration
                sample_rate = self.audio_bridge.get_source_sample_rate(self.selected_source)
                trimmed_start = start_idx / sample_rate
                trimmed_end = (len(audio_data) - end_idx - 1) / sample_rate
                
                messagebox.showinfo("Success", 
                    f"Trimmed {trimmed_start:.2f}s from start and {trimmed_end:.2f}s from end")
            else:
                messagebox.showinfo("No Change", "No silence detected at start or end")
                
        except Exception as e:
            messagebox.showerror("Error", f"Error trimming silence: {str(e)}")
            
    def add_silence(self):
        """Add silence at specified position."""
        if not self.selected_source:
            messagebox.showwarning("No Source", "Please select an audio source first")
            return
            
        try:
            position = float(self.silence_position_var.get())
            duration = float(self.silence_duration_var.get())
            
            # Get audio data and sample rate
            audio_data = self.audio_bridge.get_source_audio(self.selected_source)
            sample_rate = self.audio_bridge.get_source_sample_rate(self.selected_source)
            
            if audio_data is None or sample_rate is None:
                return
                
            # Calculate position in samples
            position_samples = int(position * sample_rate)
            silence_samples = int(duration * sample_rate)
            
            # Validate position
            if position_samples < 0:
                position_samples = 0
            elif position_samples > len(audio_data):
                position_samples = len(audio_data)
                
            # Create silence
            silence = np.zeros(silence_samples, dtype=audio_data.dtype)
            
            # Insert silence
            if position_samples == 0:
                # Add at beginning
                new_audio = np.concatenate([silence, audio_data])
            elif position_samples >= len(audio_data):
                # Add at end
                new_audio = np.concatenate([audio_data, silence])
            else:
                # Insert in middle
                new_audio = np.concatenate([
                    audio_data[:position_samples],
                    silence,
                    audio_data[position_samples:]
                ])
                
            # Update audio
            self.audio_bridge.update_source_audio(self.selected_source, new_audio)
            
            messagebox.showinfo("Success", 
                f"Added {duration}s of silence at {position}s in {self.selected_source}")
                
        except ValueError:
            messagebox.showerror("Invalid Value", "Please enter valid numbers for position and duration")
        except Exception as e:
            messagebox.showerror("Error", f"Error adding silence: {str(e)}")
            
    # State event handlers
    
    @state_aware_method
    def on_source_added(self, change):
        """Handle source added event."""
        self.refresh_sources()
        
    @state_aware_method
    def on_source_removed(self, change):
        """Handle source removed event."""
        self.refresh_sources()
        if self.selected_source == change.get('source'):
            self.selected_source = None
            self.selected_label.config(text="No source selected")
            
    @state_aware_method
    def on_source_renamed(self, change):
        """Handle source renamed event."""
        self.refresh_sources()
        
    @state_aware_method
    def on_selection_changed(self, change):
        """Handle selection change from other components."""
        if change.get('component') != self.component_name:
            source = change.get('source')
            if source in [self.source_listbox.get(i) for i in range(self.source_listbox.size())]:
                # Select in listbox
                for i in range(self.source_listbox.size()):
                    if self.source_listbox.get(i) == source:
                        self.source_listbox.selection_clear(0, tk.END)
                        self.source_listbox.selection_set(i)
                        self.source_listbox.see(i)
                        self.selected_source = source
                        self.selected_label.config(text=f"Selected: {source}")
                        break
    
    # Source preview methods
    
    def load_preview_audio(self, source_name):
        """Load audio data for preview playback."""
        try:
            # Get audio data and sample rate
            self.preview_audio_data = self.audio_bridge.get_source_audio(source_name)
            self.preview_sample_rate = self.audio_bridge.get_source_sample_rate(source_name)
            
            if self.preview_audio_data is not None and self.preview_sample_rate is not None:
                # Calculate duration
                self.audio_duration = len(self.preview_audio_data) / self.preview_sample_rate
                self.current_position = 0.0
                
                # Update UI
                self.preview_info_label.config(
                    text=f"Loaded: {source_name} ({self.audio_duration:.2f}s)"
                )
                self.update_preview_controls()
                self.draw_timeline()
                self.update_time_display()
            else:
                self.clear_preview_audio()
                
        except Exception as e:
            logger.error(f"Error loading preview audio: {e}")
            self.clear_preview_audio()
            
    def clear_preview_audio(self):
        """Clear preview audio and reset UI."""
        self.preview_audio_data = None
        self.preview_sample_rate = None
        self.audio_duration = 0.0
        self.current_position = 0.0
        self.is_playing = False
        
        if self.preview_info_label:
            self.preview_info_label.config(text="No source selected for preview")
        
        self.update_preview_controls()
        self.draw_timeline()
        self.update_time_display()
        
    def update_preview_controls(self):
        """Update preview control button states."""
        if not hasattr(self, 'play_button'):
            return
            
        has_audio = self.preview_audio_data is not None
        
        # Enable/disable buttons based on audio availability
        state = 'normal' if has_audio else 'disabled'
        self.play_button.config(state=state)
        self.pause_button.config(state=state)
        self.stop_button.config(state=state)
        self.restart_button.config(state=state)
        
        # Update play button icon and text
        if self.is_playing:
            self.play_button.config(text=" Playing", image=self.pause_icon)
        else:
            self.play_button.config(text=" Play", image=self.play_icon)
            
    def draw_timeline(self):
        """Draw the timeline with position indicator."""
        if not self.preview_canvas:
            return
            
        # Clear canvas
        self.preview_canvas.delete("all")
        
        # Get canvas dimensions
        width = self.preview_canvas.winfo_width()
        height = self.preview_canvas.winfo_height()
        
        if width <= 1:  # Canvas not ready yet
            return
            
        # Draw beautiful timeline track
        track_y = height // 2
        self.preview_canvas.create_rectangle(
            10, track_y - 3, width - 10, track_y + 3,
            fill=theme.SLIDER_TRACK,
            outline=theme.BORDER,
            width=1
        )
        
        if self.audio_duration > 0:
            # Calculate position
            progress = self.current_position / self.audio_duration
            x_pos = 10 + progress * (width - 20)
            
            # Draw progress fill (played portion in blue)
            track_y = height // 2
            if x_pos > 10:
                self.preview_canvas.create_rectangle(
                    10, track_y - 3, x_pos, track_y + 3,
                    fill=theme.ACCENT_PRIMARY,
                    outline='',
                    width=0
                )

            # Draw playhead circle (modern style)
            circle_radius = 6
            self.preview_canvas.create_oval(
                x_pos - circle_radius, track_y - circle_radius,
                x_pos + circle_radius, track_y + circle_radius,
                fill=theme.ACCENT_PRIMARY,
                outline='#FFFFFF',
                width=2
            )
            
            # Draw time markers (every second)
            track_y = height // 2
            for i in range(int(self.audio_duration) + 1):
                x = 10 + (i / self.audio_duration) * (width - 20)
                # Time tick marks
                self.preview_canvas.create_line(
                    x, track_y + 5, x, track_y + 10,
                    fill=theme.TEXT_SECONDARY,
                    width=1
                )

                # Add second labels (every 5 seconds or if short duration)
                if i % 5 == 0 or self.audio_duration < 10:
                    self.preview_canvas.create_text(
                        x, height - 8, text=f"{i}s",
                        font=theme.FONT_SMALL,
                        fill=theme.TEXT_SECONDARY
                    )
                    
    def update_time_display(self):
        """Update the time display label."""
        if hasattr(self, 'time_label'):
            time_text = f"{self.current_position:.2f}s / {self.audio_duration:.2f}s"
            self.time_label.config(text=time_text)
            
    def on_timeline_click(self, event):
        """Handle timeline click for seeking."""
        if self.preview_audio_data is None or self.audio_duration <= 0:
            return
            
        # Get click position
        width = self.preview_canvas.winfo_width()
        click_x = event.x
        
        # Calculate time position
        if width > 20:
            progress = max(0, min(1, (click_x - 10) / (width - 20)))
            self.current_position = progress * self.audio_duration
            
            # Update display
            self.draw_timeline()
            self.update_time_display()
            
            # Seek audio engine if currently playing
            if self.is_playing and self.audio_engine and hasattr(self.audio_engine, 'seek_source_preview'):
                self.audio_engine.seek_source_preview(self.current_position)
            
    def on_canvas_resize(self, event):
        """Handle canvas resize to redraw timeline."""
        self.draw_timeline()

    # Preview playback methods (fully implemented with audio engine integration)
    
    def preview_play(self):
        """Start or resume preview playback."""
        if self.preview_audio_data is None or not self.selected_source:
            return
            
        self.is_playing = True
        self.update_preview_controls()
        
        # Use audio engine for actual playback if available
        if self.audio_engine and hasattr(self.audio_engine, 'start_source_preview'):
            success = self.audio_engine.start_source_preview(self.selected_source, self.current_position)
            if success:
                logger.debug(f"Playing {self.selected_source} from {self.current_position:.2f}s")
                # Start position update timer
                self.start_position_updates()
            else:
                self.is_playing = False
                self.update_preview_controls()
        else:
            # Fallback to simulation
            logger.debug(f"[Simulated] Playing {self.selected_source} from {self.current_position:.2f}s")
        
    def preview_pause(self):
        """Pause preview playback."""
        self.is_playing = False
        self.update_preview_controls()
        
        # Pause audio engine if available
        if self.audio_engine and hasattr(self.audio_engine, 'pause_source_preview'):
            self.audio_engine.pause_source_preview()
            
        # Stop position updates
        self.stop_position_updates()
        logger.debug(f"Paused at {self.current_position:.2f}s")
        
    def preview_stop(self):
        """Stop preview playback and reset position."""
        self.is_playing = False
        self.current_position = 0.0
        self.update_preview_controls()
        self.draw_timeline()
        self.update_time_display()
        
        # Stop audio engine if available
        if self.audio_engine and hasattr(self.audio_engine, 'stop_source_preview'):
            self.audio_engine.stop_source_preview()
            
        # Stop position updates
        self.stop_position_updates()
        logger.debug("Stopped and reset to beginning")
        
    def preview_restart(self):
        """Restart preview playback from beginning."""
        was_playing = self.is_playing
        
        # Stop current playback
        self.preview_stop()
        
        # Start from beginning if was playing
        if was_playing:
            self.preview_play()
        else:
            logger.debug("Reset to beginning")
            
    def start_position_updates(self):
        """Start timer for updating playback position."""
        if hasattr(self, 'position_timer') and self.position_timer:
            self.content_frame.after_cancel(self.position_timer)
            
        self.update_playback_position()
        
    def stop_position_updates(self):
        """Stop position update timer."""
        if hasattr(self, 'position_timer') and self.position_timer:
            self.content_frame.after_cancel(self.position_timer)
            self.position_timer = None
            
    def update_playback_position(self):
        """Update current position during playback."""
        if not self.is_playing:
            return
            
        # Update position (rough estimate - could be improved with actual audio engine feedback)
        self.current_position += 0.1  # 100ms updates
        
        # Check if we've reached the end
        if self.current_position >= self.audio_duration:
            self.current_position = self.audio_duration
            self.is_playing = False
            self.update_preview_controls()
            self.stop_position_updates()
            logger.debug("Playback completed")
        else:
            # Update display
            self.draw_timeline()
            self.update_time_display()
            
            # Schedule next update
            self.position_timer = self.content_frame.after(100, self.update_playback_position)
    
    # Playback control methods removed - single source editing focus