"""
Modular Main Window - Completely customizable workspace

Uses dockable panels that can be shown/hidden, resized, and positioned
anywhere. Provides a Photoshop-style interface where users control
their workspace layout completely.
"""

import logging
import tkinter as tk
from tkinter import ttk, filedialog, messagebox
import sys
from pathlib import Path

# Set up logger
logger = logging.getLogger(__name__)

# Import dockable panel system
from .dockable_panel import PanelManager
from .panels.audio_sources_panel import AudioSourcesPanel
from .panels.source_properties_panel import SourcePropertiesPanel
from .panels.spatial_player import SpatialPlayer
from .panels.transport_panel import TransportPanel
from .panels.audio_processing_panel import AudioProcessingPanel

# Import theme and error dialogs
from . import theme
from .icons import get_icon
from . import error_dialogs

# Import existing components
sys.path.insert(0, str(Path(__file__).parent.parent))
from integration.sampler_bridge import SamplerBridge
from core.audio_engine import AudioEngine
from core.project_manager import ProjectManager
from core.state_manager import state_manager
from .spatial_view import SpatialView
from .export_dialog import ExportDialog


class ModularMainWindow:
    """Main application window with completely modular interface."""

    def __init__(self):
        """Initialize the modular main window with professional theme."""
        # Try to use TkinterDnD if available for drag-and-drop support
        try:
            from tkinterdnd2 import TkinterDnD
            self.root = TkinterDnD.Tk()
            logger.info("Using TkinterDnD root window for drag-and-drop support")
        except ImportError:
            self.root = tk.Tk()
            logger.info("Using standard Tk root window (drag-and-drop limited)")

        self.root.title("SHAC Studio v2.0")
        self.root.geometry("1024x768")

        # Set application icon
        try:
            icon_path = Path(__file__).parent.parent / "assets" / "shac-icon.png"
            if icon_path.exists():
                icon = tk.PhotoImage(file=str(icon_path))
                self.root.iconphoto(True, icon)
                logger.info(f"Application icon set from {icon_path}")
            else:
                logger.warning(f"Icon file not found at {icon_path}")
        except Exception as e:
            logger.warning(f"Could not set application icon: {e}")

        # Apply professional dark theme to root window
        self.root.config(bg=theme.BACKGROUND_PRIMARY)

        # Configure ttk theme
        style = ttk.Style()
        theme.Theme.configure_ttk_style(style)
        
        # Initialize audio components
        self.audio_bridge = SamplerBridge()
        self.audio_engine = AudioEngine()
        self.project_manager = ProjectManager()
        
        # Give audio bridge reference to main window for coordination
        self.audio_bridge.main_window = self
        
        # Set up audio engine callback for position updates
        self.audio_engine.set_position_callback(self.on_playback_position_update)
        
        # Create spatial view (still needed as core component)
        self.spatial_view = SpatialView(self.root)
        self.spatial_view.audio_engine = self.audio_engine
        self.spatial_view.parent = self  # Give spatial view access to main window
        
        # Connect audio bridge to spatial view for source positioning
        self.audio_bridge.set_spatial_view(self.spatial_view)
        
        # Initialize panel manager
        self.panel_manager = PanelManager(self)
        
        # Initialize all dockable panels
        self.setup_panels()
        
        # Set up minimal main window UI
        self.setup_ui()
        
        # Set up default panel layout
        self.setup_default_layout()
        
    def setup_panels(self):
        """Initialize panel registry for lazy loading."""
        # Initialize empty panel references
        self.audio_sources_panel = None
        self.source_properties_panel = None
        self.spatial_player_panel = None
        self.transport_panel = None
        self.audio_processing_panel = None
        
        logger.info("Panel system initialized - panels will load on demand")
        
    def setup_ui(self):
        """Set up the minimal main window UI."""
        # Main window is now just menu bar - everything else is modular!
        self.root.grid_rowconfigure(0, weight=1)
        self.root.grid_columnconfigure(0, weight=1)
        
        # Create an empty workspace area with dark theme
        workspace_frame = tk.Frame(self.root, bg=theme.BACKGROUND_PRIMARY)
        workspace_frame.pack(fill='both', expand=True)

        # Create welcome content with dark theme
        welcome_frame = tk.Frame(workspace_frame, bg=theme.BACKGROUND_PRIMARY)
        welcome_frame.place(relx=0.5, rely=0.5, anchor='center')

        # Title - use accent blue for the main title!
        title_label = tk.Label(
            welcome_frame,
            text="SHAC Studio",
            font=theme.FONT_TITLE,
            fg=theme.TEXT_ACCENT,  # Beautiful blue for title
            bg=theme.BACKGROUND_PRIMARY
        )
        title_label.pack(pady=(0, 5))

        # Subtitle - light text
        subtitle_label = tk.Label(
            welcome_frame,
            text="Revolutionary Spatial Audio Creation",
            font=theme.FONT_LARGE,
            fg=theme.TEXT_PRIMARY,
            bg=theme.BACKGROUND_PRIMARY
        )
        subtitle_label.pack(pady=(0, 20))

        # Version info
        version_label = tk.Label(
            welcome_frame,
            text="v2.0",
            font=theme.FONT_SMALL,
            fg=theme.TEXT_SECONDARY,
            bg=theme.BACKGROUND_PRIMARY
        )
        version_label.pack(pady=(0, 30))

        # Quick start guide with dark theme
        guide_frame = tk.LabelFrame(
            welcome_frame,
            text="Quick Start",
            font=theme.FONT_NORMAL,
            fg=theme.TEXT_ACCENT,  # Blue for section header
            bg=theme.BACKGROUND_SECONDARY,
            padx=20,
            pady=20
        )
        guide_frame.pack()

        steps = [
            "1. Load audio files: View → Audio Sources",
            "2. Position sources: View → Spatial Player",
            "3. Experience spatial audio with WASD controls",
            "4. Export to SHAC: File → Export to SHAC"
        ]

        for step in steps:
            step_label = tk.Label(
                guide_frame,
                text=step,
                font=theme.FONT_NORMAL,
                fg=theme.TEXT_PRIMARY,  # White/light text for readability
                bg=theme.BACKGROUND_SECONDARY,
                anchor='w'
            )
            step_label.pack(anchor='w', pady=2, fill='x')

        # Pro tip with accent blue
        tip_label = tk.Label(
            welcome_frame,
            text="💡 Press F1 for help at any time",
            font=theme.FONT_NORMAL,
            fg=theme.TEXT_ACCENT,  # Blue for emphasis
            bg=theme.BACKGROUND_PRIMARY
        )
        tip_label.pack(pady=(20, 0))
        
        # Menu bar
        self.create_menu()

        # Initialize Edit menu state
        self.update_edit_menu()

        # Set up keyboard shortcuts
        self.setup_keyboard_shortcuts()

        # Subscribe to undoable events to update Edit menu
        self.subscribe_to_undoable_events()
        
        # Status bar
        self.setup_status_bar()
        
    def create_menu(self):
        """Create the application menu with panel controls and professional theme."""
        menubar = tk.Menu(
            self.root,
            bg=theme.BACKGROUND_TERTIARY,
            fg=theme.TEXT_PRIMARY,
            activebackground=theme.ACCENT_PRIMARY,
            activeforeground='#FFFFFF',
            borderwidth=0
        )
        self.root.config(menu=menubar)
        
        # File menu with theme
        file_menu = tk.Menu(
            menubar,
            tearoff=0,
            bg=theme.BACKGROUND_SECONDARY,
            fg=theme.TEXT_PRIMARY,
            activebackground=theme.ACCENT_PRIMARY,
            activeforeground='#FFFFFF',
            borderwidth=0
        )
        menubar.add_cascade(label="File", menu=file_menu)
        file_menu.add_command(label="New Project", command=self.new_project, accelerator="Ctrl+N")
        file_menu.add_command(label="Open Project...", command=self.open_project, accelerator="Ctrl+O")
        file_menu.add_command(label="Save Project", command=self.save_project, accelerator="Ctrl+S")
        file_menu.add_command(label="Save Project As...", command=self.save_project_as, accelerator="Ctrl+Shift+S")
        file_menu.add_separator()
        
        # Recent projects submenu with theme
        self.recent_menu = tk.Menu(
            file_menu,
            tearoff=0,
            bg=theme.BACKGROUND_SECONDARY,
            fg=theme.TEXT_PRIMARY,
            activebackground=theme.ACCENT_PRIMARY,
            activeforeground='#FFFFFF'
        )
        file_menu.add_cascade(label="Recent Projects", menu=self.recent_menu)
        self._update_recent_projects_menu()
        
        file_menu.add_separator()
        file_menu.add_command(label="Export to SHAC...", command=self.export_shac, accelerator="Ctrl+E")
        file_menu.add_separator()
        file_menu.add_command(label="Exit", command=self.root.quit)

        # Edit menu with theme
        edit_menu = tk.Menu(
            menubar,
            tearoff=0,
            bg=theme.BACKGROUND_SECONDARY,
            fg=theme.TEXT_PRIMARY,
            activebackground=theme.ACCENT_PRIMARY,
            activeforeground='#FFFFFF',
            borderwidth=0
        )
        menubar.add_cascade(label="Edit", menu=edit_menu)
        edit_menu.add_command(label="Undo", command=self.undo, accelerator="Ctrl+Z")
        edit_menu.add_command(label="Redo", command=self.redo, accelerator="Ctrl+Shift+Z")

        # Store edit menu for updating states
        self.edit_menu = edit_menu

        # View menu - The heart of the modular system! (themed)
        view_menu = tk.Menu(
            menubar,
            tearoff=0,
            bg=theme.BACKGROUND_SECONDARY,
            fg=theme.TEXT_PRIMARY,
            activebackground=theme.ACCENT_PRIMARY,
            activeforeground='#FFFFFF',
            borderwidth=0
        )
        menubar.add_cascade(label="View", menu=view_menu)
        
        # Panel visibility toggles with lazy loading
        view_menu.add_checkbutton(
            label="Audio Sources",
            command=lambda: self.lazy_toggle_panel("audio_sources"),
            variable=self.get_panel_visibility_var("audio_sources")
        )
        view_menu.add_checkbutton(
            label="Spatial Positioning", 
            command=lambda: self.lazy_toggle_panel("source_properties"),
            variable=self.get_panel_visibility_var("source_properties")
        )
        view_menu.add_checkbutton(
            label="Spatial Audio Player",
            command=lambda: self.lazy_toggle_panel("spatial_player"),
            variable=self.get_panel_visibility_var("spatial_player")
        )
        view_menu.add_checkbutton(
            label="Audio Processing",
            command=lambda: self.lazy_toggle_panel("audio_processing"),
            variable=self.get_panel_visibility_var("audio_processing")
        )
        
        view_menu.add_separator()
        
        # Layout presets
        layout_menu = tk.Menu(
            view_menu,
            tearoff=0,
            bg=theme.BACKGROUND_SECONDARY,
            fg=theme.TEXT_PRIMARY,
            activebackground=theme.ACCENT_PRIMARY,
            activeforeground='#FFFFFF'
        )
        view_menu.add_cascade(label="Layout Presets", menu=layout_menu)
        
        layout_menu.add_command(label="Default Layout", command=self.setup_default_layout)
        layout_menu.add_command(label="Compact Layout", command=self.setup_compact_layout)
        layout_menu.add_command(label="Wide Layout", command=self.setup_wide_layout)
        layout_menu.add_separator()
        layout_menu.add_command(label="Hide All Panels", command=self.hide_all_panels)
        layout_menu.add_command(label="Show All Panels", command=self.show_all_panels)
        
        # Window menu
        window_menu = tk.Menu(
            menubar,
            tearoff=0,
            bg=theme.BACKGROUND_SECONDARY,
            fg=theme.TEXT_PRIMARY,
            activebackground=theme.ACCENT_PRIMARY,
            activeforeground='#FFFFFF'
        )
        menubar.add_cascade(label="Window", menu=window_menu)
        window_menu.add_command(label="Reset Panel Positions", command=self.reset_panel_positions)
        window_menu.add_command(label="Save Layout", command=self.panel_manager.save_panel_states)
        
        # Help menu
        help_menu = tk.Menu(
            menubar,
            tearoff=0,
            bg=theme.BACKGROUND_SECONDARY,
            fg=theme.TEXT_PRIMARY,
            activebackground=theme.ACCENT_PRIMARY,
            activeforeground='#FFFFFF'
        )
        menubar.add_cascade(label="Help", menu=help_menu)
        help_menu.add_command(label="Getting Started", command=self.show_help)
        help_menu.add_command(label="About SHAC Studio", command=self.show_about)
        
    def setup_keyboard_shortcuts(self):
        """Set up keyboard shortcuts for the application."""
        # Edit shortcuts
        self.root.bind('<Control-z>', lambda e: self.undo())
        self.root.bind('<Control-Z>', lambda e: self.undo())
        self.root.bind('<Control-Shift-z>', lambda e: self.redo())
        self.root.bind('<Control-Shift-Z>', lambda e: self.redo())

        # File shortcuts
        self.root.bind('<Control-n>', lambda e: self.new_project())
        self.root.bind('<Control-N>', lambda e: self.new_project())
        self.root.bind('<Control-o>', lambda e: self.open_project())
        self.root.bind('<Control-O>', lambda e: self.open_project())
        self.root.bind('<Control-s>', lambda e: self.save_project())
        self.root.bind('<Control-S>', lambda e: self.save_project())
        self.root.bind('<Control-Shift-s>', lambda e: self.save_project_as())
        self.root.bind('<Control-Shift-S>', lambda e: self.save_project_as())
        self.root.bind('<Control-e>', lambda e: self.export_shac())
        self.root.bind('<Control-E>', lambda e: self.export_shac())

        # Transport shortcuts
        self.root.bind('<space>', lambda e: self.toggle_playback())
        self.root.bind('<r>', lambda e: self.stop_audio())
        self.root.bind('<R>', lambda e: self.stop_audio())

        # Help shortcuts
        self.root.bind('<F1>', lambda e: self.show_help())
        
    def setup_status_bar(self):
        """Create status bar at bottom of window."""
        self.status_frame = ttk.Frame(self.root)
        self.status_frame.pack(side='bottom', fill='x')
        
        # Status label
        self.status_label = ttk.Label(
            self.status_frame,
            text="Ready",
            relief='sunken',
            anchor='w'
        )
        self.status_label.pack(side='left', fill='x', expand=True, padx=2, pady=2)
        
        # Audio format label
        self.format_label = ttk.Label(
            self.status_frame,
            text="",
            relief='sunken',
            anchor='e'
        )
        self.format_label.pack(side='right', padx=2, pady=2)
        
        # Source count label
        self.source_count_label = ttk.Label(
            self.status_frame,
            text="0 sources",
            relief='sunken',
            anchor='e'
        )
        self.source_count_label.pack(side='right', padx=2, pady=2)
        
    def update_status(self, message: str):
        """Update status bar message."""
        if hasattr(self, 'status_label'):
            self.status_label.config(text=message)
            # Clear status after 3 seconds
            self.root.after(3000, lambda: self.status_label.config(text="Ready"))
            
    def update_source_count(self):
        """Update source count in status bar."""
        if hasattr(self, 'source_count_label'):
            count = len(self.audio_engine.sources)
            text = f"{count} source{'s' if count != 1 else ''}"
            self.source_count_label.config(text=text)
        
    def get_panel_visibility_var(self, panel_id: str):
        """Get visibility variable for menu checkbutton."""
        if not hasattr(self, '_panel_vars'):
            self._panel_vars = {}
        
        if panel_id not in self._panel_vars:
            self._panel_vars[panel_id] = tk.BooleanVar()
            
        # Update variable to match actual panel state (checked = visible)
        self._panel_vars[panel_id].set(self.panel_manager.is_panel_visible(panel_id))
        return self._panel_vars[panel_id]
        
    def lazy_toggle_panel(self, panel_id: str):
        """Toggle panel with lazy loading."""
        # Create panel if it doesn't exist
        panel_was_created = not self.is_panel_created(panel_id)
        if panel_was_created:
            self.create_panel(panel_id)
            # For newly created panels, always show them (don't toggle)
            # This avoids race conditions with saved state
            self.panel_manager.show_panel(panel_id)
        else:
            # For existing panels, toggle normally
            self.panel_manager.toggle_panel(panel_id)

        # Update checkbox
        if hasattr(self, '_panel_vars') and panel_id in self._panel_vars:
            self._panel_vars[panel_id].set(self.panel_manager.is_panel_visible(panel_id))
            
    def is_panel_created(self, panel_id: str) -> bool:
        """Check if panel has been created."""
        panel_attr = f"{panel_id}_panel"
        return hasattr(self, panel_attr) and getattr(self, panel_attr) is not None
        
    def create_panel(self, panel_id: str):
        """Create a specific panel on demand."""
        logger.debug(f"Loading {panel_id} panel...")
        
        if panel_id == "audio_sources":
            self.audio_sources_panel = AudioSourcesPanel(self.audio_bridge, self.root)
            self.audio_sources_panel.set_spatial_view(self.spatial_view)
            self.panel_manager.register_panel("audio_sources", self.audio_sources_panel)
            
        elif panel_id == "source_properties":
            self.source_properties_panel = SourcePropertiesPanel(self.spatial_view, self)
            self.source_properties_panel.tk_parent = self.root
            self.source_properties_panel.audio_engine = self.audio_engine
            self.source_properties_panel.parent_window = self
            self.panel_manager.register_panel("source_properties", self.source_properties_panel)
            
        elif panel_id == "spatial_player":
            self.spatial_player_panel = SpatialPlayer(self.audio_engine, self.root)
            # Give spatial player access to the main window for audio bridge
            self.spatial_player_panel.parent_window = self
            # Override the parent_window for Tkinter operations to use the root
            self.spatial_player_panel.tk_parent = self.root
            self.panel_manager.register_panel("spatial_player", self.spatial_player_panel)
            # Initialize sources if they already exist
            self.spatial_player_panel.refresh_sources()
            
        elif panel_id == "transport":
            self.transport_panel = TransportPanel(self.root)
            self.panel_manager.register_panel("transport", self.transport_panel)
            # Setup transport controls to use audio engine
            self.setup_transport()
            
        elif panel_id == "audio_processing":
            self.audio_processing_panel = AudioProcessingPanel(self.audio_bridge, self.root)
            self.audio_processing_panel.audio_engine = self.audio_engine  # Connect audio engine
            self.panel_manager.register_panel("audio_processing", self.audio_processing_panel)
            # Initialize source list if sources already exist
            self.audio_processing_panel.refresh_sources()
            
        logger.debug(f"{panel_id} panel loaded and ready")
        
    def toggle_panel_and_refresh(self, panel_id: str):
        """Toggle panel and refresh checkbox state."""
        self.panel_manager.toggle_panel(panel_id)
        if hasattr(self, '_panel_vars') and panel_id in self._panel_vars:
            self._panel_vars[panel_id].set(self.panel_manager.is_panel_visible(panel_id))
        
    def setup_default_layout(self):
        """Set up the default panel layout - essential panels for typical workflow."""
        logger.info("Setting up default layout")

        # Show most commonly used panels
        self.panel_manager.show_panel("audio_sources")
        self.panel_manager.show_panel("spatial_player")
        self.panel_manager.show_panel("transport")
        self.panel_manager.show_panel("source_properties")

        # Hide optional panels
        self.panel_manager.hide_panel("audio_processing")

        # Set sensible positions for default panels
        if "audio_sources" in self.panel_manager.panels:
            self.panel_manager.panels["audio_sources"].set_geometry(50, 100, 350, 500)
        if "spatial_player" in self.panel_manager.panels:
            self.panel_manager.panels["spatial_player"].set_geometry(420, 100, 700, 600)
        if "source_properties" in self.panel_manager.panels:
            self.panel_manager.panels["source_properties"].set_geometry(1140, 100, 320, 400)
        if "transport" in self.panel_manager.panels:
            self.panel_manager.panels["transport"].set_geometry(420, 720, 700, 150)

    def setup_compact_layout(self):
        """Set up a compact layout for smaller screens - minimal panels only."""
        logger.info("Setting up compact layout")

        # Show only essential panels for basic workflow
        self.panel_manager.show_panel("audio_sources")
        self.panel_manager.show_panel("spatial_player")
        self.panel_manager.show_panel("transport")

        # Hide everything else
        self.panel_manager.hide_panel("source_properties")
        self.panel_manager.hide_panel("audio_processing")

        # Compact positioning - smaller, tighter arrangement
        if "audio_sources" in self.panel_manager.panels:
            self.panel_manager.panels["audio_sources"].set_geometry(50, 100, 300, 450)
        if "spatial_player" in self.panel_manager.panels:
            self.panel_manager.panels["spatial_player"].set_geometry(370, 100, 600, 500)
        if "transport" in self.panel_manager.panels:
            self.panel_manager.panels["transport"].set_geometry(370, 620, 600, 130)
        
    def setup_wide_layout(self):
        """Set up a wide layout for large monitors."""
        # Spread panels across wide screen
        self.show_all_panels()
        
        if "spatial_view" in self.panel_manager.panels:
            self.panel_manager.panels["spatial_view"].set_geometry(600, 100, 800, 600)
        if "audio_sources" in self.panel_manager.panels:
            self.panel_manager.panels["audio_sources"].set_geometry(50, 100, 350, 600)
        if "source_properties" in self.panel_manager.panels:
            self.panel_manager.panels["source_properties"].set_geometry(1420, 100, 300, 400)
        if "audio_processing" in self.panel_manager.panels:
            self.panel_manager.panels["audio_processing"].set_geometry(1740, 100, 450, 500)
            
    def show_all_panels(self):
        """Show all panels."""
        for panel_id in self.panel_manager.panels:
            self.panel_manager.show_panel(panel_id)
            
    def hide_all_panels(self):
        """Hide all panels."""
        for panel_id in self.panel_manager.panels:
            self.panel_manager.hide_panel(panel_id)
            
    def reset_panel_positions(self):
        """Reset all panels to default positions."""
        for panel in self.panel_manager.panels.values():
            panel.set_geometry(panel.default_x, panel.default_y, 
                             panel.default_width, panel.default_height)
    
    # Transport control methods (now using modular transport panel)
    def setup_transport(self):
        """Setup transport controls to use audio engine."""
        if self.transport_panel:
            self.transport_panel.play = self.play_audio
            self.transport_panel.pause = self.pause_audio
            self.transport_panel.stop = self.stop_audio
            self.transport_panel.on_timeline_change = self.seek_audio
        
    def play_audio(self):
        """Start audio playback."""
        if self.transport_panel:
            self.transport_panel.is_playing = True
            self.transport_panel.play_button.config(
                text=" Pause",
                image=self.transport_panel.pause_icon,
                bg=theme.SUCCESS
            )
        self.audio_engine.play()

    def pause_audio(self):
        """Pause audio playback."""
        if self.transport_panel:
            self.transport_panel.is_playing = False
            self.transport_panel.play_button.config(
                text=" Play",
                image=self.transport_panel.play_icon,
                bg=theme.BUTTON_NORMAL
            )
        self.audio_engine.pause()

    def stop_audio(self):
        """Stop audio playback."""
        if self.transport_panel:
            self.transport_panel.is_playing = False
            self.transport_panel.play_button.config(
                text=" Play",
                image=self.transport_panel.play_icon,
                bg=theme.BUTTON_NORMAL
            )
            self.transport_panel.timeline_var.set(0)
            self.transport_panel.update_time_display()
        self.audio_engine.stop()
        
    def seek_audio(self, value):
        """Seek to position in audio."""
        if self.transport_panel and self.transport_panel.total_time > 0:
            position = float(value) / 100.0
            time_position = position * self.transport_panel.total_time
            self.audio_engine.seek(time_position)
            self.transport_panel.current_time = time_position
            self.transport_panel.update_time_display()
    
    def toggle_playback(self):
        """Toggle play/pause state (keyboard shortcut)."""
        if hasattr(self, 'transport_panel') and self.transport_panel:
            if self.transport_panel.is_playing:
                self.stop_audio()
            else:
                self.play_audio()

    # Edit methods
    def subscribe_to_undoable_events(self):
        """Subscribe to state events that affect undo/redo availability."""
        from core.state_manager import StateEvent

        # Subscribe to all undoable events
        undoable_events = [
            StateEvent.SOURCE_POSITION_CHANGED,
            StateEvent.SOURCE_ADDED,
            StateEvent.SOURCE_REMOVED,
            StateEvent.SOURCE_RENAMED,
            StateEvent.SOURCE_MUTED,
            StateEvent.SOURCE_UNMUTED,
            StateEvent.AUDIO_DATA_EDITED,
        ]

        for event in undoable_events:
            state_manager.subscribe(event, self._on_undoable_event, "MainWindow")

    def _on_undoable_event(self, change):
        """Handle undoable state changes by updating Edit menu."""
        # Use after_idle to avoid blocking the event handler
        self.root.after_idle(self.update_edit_menu)

    def update_edit_menu(self):
        """Update Edit menu with current undo/redo states."""
        if not hasattr(self, 'edit_menu'):
            return

        # Update Undo menu item (index 0)
        if state_manager.can_undo():
            undo_desc = state_manager.get_undo_description()
            self.edit_menu.entryconfigure(0, label=f"Undo {undo_desc}", state='normal')
        else:
            self.edit_menu.entryconfigure(0, label="Undo", state='disabled')

        # Update Redo menu item (index 1)
        if state_manager.can_redo():
            redo_desc = state_manager.get_redo_description()
            self.edit_menu.entryconfigure(1, label=f"Redo {redo_desc}", state='normal')
        else:
            self.edit_menu.entryconfigure(1, label="Redo", state='disabled')

    def undo(self):
        """Undo the last action."""
        if state_manager.undo():
            desc = state_manager.get_undo_description() or "Action"
            logger.info(f"Undid: {desc}")
            self.update_edit_menu()
        else:
            logger.debug("Nothing to undo")

    def redo(self):
        """Redo the last undone action."""
        if state_manager.redo():
            desc = state_manager.get_redo_description() or "Action"
            logger.info(f"Redid: {desc}")
            self.update_edit_menu()
        else:
            logger.debug("Nothing to redo")

    # Project management methods
    def new_project(self):
        """Create a new project."""
        # Check if current project has unsaved changes
        if self.project_manager.is_project_modified():
            response = messagebox.askyesnocancel(
                "Unsaved Changes",
                "Save changes to current project before creating new project?",
                parent=self.root
            )
            if response is None:  # Cancel
                return
            elif response:  # Yes, save
                if not self.save_project():
                    return  # Save failed or was cancelled
        
        # Create new project
        self.project_manager.new_project()
        self.audio_engine.clear_sources()
        
        # Clear UI components
        self._clear_ui_for_new_project()
        
        # Update window title
        self._update_window_title()
        
        logger.info("New project created")
        
    def open_project(self):
        """Open an existing project."""
        # Check if current project has unsaved changes
        if self.project_manager.is_project_modified():
            response = messagebox.askyesnocancel(
                "Unsaved Changes",
                "Save changes to current project before opening another?",
                parent=self.root
            )
            if response is None:  # Cancel
                return
            elif response:  # Yes, save
                if not self.save_project():
                    return  # Save failed or was cancelled
        
        # Show file dialog
        filepath = filedialog.askopenfilename(
            title="Open SHAC Project",
            defaultextension=".shacproject",
            filetypes=[
                ("SHAC Project files", "*.shacproject"),
                ("All files", "*.*")
            ],
            parent=self.root
        )
        
        if filepath:
            if self.project_manager.load_project(filepath):
                self._load_project_into_ui()
                self._update_window_title()
                self._update_recent_projects_menu()
                error_dialogs.show_success("Project Loaded", f"Successfully loaded: {self.project_manager.get_project_name()}", parent=self.root)
            else:
                error_dialogs.show_error("Load Error", "Failed to load project file.\n\nThe file may be corrupted or in an incompatible format.", parent=self.root)
        
    def save_project(self):
        """Save the current project."""
        if self.project_manager.current_project.filepath:
            # Save to existing file
            self._collect_project_data()
            success = self.project_manager.save_project()
            if success:
                self._update_window_title()
                self._update_recent_projects_menu()
                return True
            else:
                error_dialogs.show_error("Save Error", "Failed to save project.\n\nPlease check that you have write permissions for this location.", parent=self.root)
                return False
        else:
            # No file path - use Save As
            return self.save_project_as()
    
    def save_project_as(self):
        """Save project with new filename."""
        filepath = filedialog.asksaveasfilename(
            title="Save SHAC Project As",
            defaultextension=".shacproject",
            filetypes=[
                ("SHAC Project files", "*.shacproject"),
                ("All files", "*.*")
            ],
            parent=self.root
        )
        
        if filepath:
            self._collect_project_data()
            success = self.project_manager.save_as_project(filepath)
            if success:
                self._update_window_title()
                self._update_recent_projects_menu()
                error_dialogs.show_success("Project Saved", f"Successfully saved: {self.project_manager.get_project_name()}", parent=self.root)
                return True
            else:
                error_dialogs.show_error("Save Error", "Failed to save project.\n\nPlease check that you have write permissions for this location.", parent=self.root)
                return False
        return False
    
    def _collect_project_data(self):
        """Collect current state into project manager."""
        # Clear existing sources in project
        self.project_manager.current_project.sources = []
        
        # Add current audio sources
        if hasattr(self, 'audio_engine') and self.audio_engine.sources:
            for source_id, source in self.audio_engine.sources.items():
                # Get filepath from sampler bridge's loaded_samples
                sample_info = self.audio_bridge.loaded_samples.get(source_id, {})
                filepath = sample_info.get('filepath', '')
                name = sample_info.get('name', source_id)
                
                self.project_manager.current_project.add_source(
                    source_id=source.source_id,
                    name=name,
                    filepath=filepath,
                    position=tuple(source.position),
                    volume=source.volume,
                    muted=source.is_muted
                )
    
    def _load_project_into_ui(self):
        """Load project data into UI components."""
        logger.info("Loading project into UI...")

        # Show loading dialog for multi-source projects
        loading_dialog = None
        num_sources = len(self.project_manager.current_project.sources)
        logger.info(f"Project has {num_sources} sources")

        if num_sources > 3:
            loading_dialog = error_dialogs.LoadingDialog(
                self.root,
                title="Loading Project",
                message=f"Loading project with {num_sources} audio sources..."
            )

        try:
            # Clear current state - remove all sources
            if hasattr(self, 'audio_engine') and self.audio_engine.sources:
                # Clear sources from audio engine
                self.audio_engine.sources.clear()

            # Clear sources from sampler bridge
            if hasattr(self, 'audio_bridge'):
                self.audio_bridge.loaded_samples.clear()

            # Clear sources from audio sources panel if it exists
            if hasattr(self, 'audio_sources_panel') and self.audio_sources_panel:
                self.audio_sources_panel.clear_sources()

            # Load sources from project
            for i, project_source in enumerate(self.project_manager.current_project.sources):
                logger.info(f"Loading source {i+1}/{num_sources}: {project_source.name} from {project_source.filepath}")

                if loading_dialog:
                    loading_dialog.update_message(
                        f"Loading source {i+1} of {num_sources}...\n{project_source.name}"
                    )
                try:
                    # Check if file exists
                    if not Path(project_source.filepath).exists():
                        logger.warning(f"File not found: {project_source.filepath}")
                        response = messagebox.askyesnocancel(
                            "Missing Audio File",
                            f"Cannot find audio file: {project_source.filepath}\n\nWould you like to locate it?",
                            parent=self.root
                        )
                        if response is None:  # Cancel loading
                            return
                        elif response:  # Yes, locate file
                            new_filepath = filedialog.askopenfilename(
                                title=f"Locate: {Path(project_source.filepath).name}",
                                filetypes=[
                                    ("Audio files", "*.wav *.mp3 *.flac *.ogg"),
                                    ("All files", "*.*")
                                ],
                                parent=self.root
                            )
                            if new_filepath:
                                project_source.filepath = new_filepath
                            else:
                                continue  # Skip this source
                        else:  # No, skip this source
                            continue

                    # Load audio file through sampler bridge
                    loaded_source_id = self.audio_bridge.add_audio_to_project(
                        filepath=project_source.filepath,
                        position=project_source.position,
                        name=project_source.name
                    )

                    logger.info(f"Source loaded with ID: {loaded_source_id}")

                    # Set properties if successfully loaded
                    if loaded_source_id and loaded_source_id in self.audio_engine.sources:
                        logger.info(f"Setting properties for source {loaded_source_id}")
                        source = self.audio_engine.sources[loaded_source_id]
                        source.volume = project_source.volume
                        source.is_muted = project_source.muted

                        # Update the audio sources panel if it exists
                        if hasattr(self, 'audio_sources_panel') and self.audio_sources_panel:
                            self.audio_sources_panel.refresh_sources()

                except Exception as e:
                    error_dialogs.handle_exception(
                        f"load audio source '{project_source.name}'",
                        e,
                        parent=self.root
                    )

            # Refresh UI components
            logger.info("Refreshing UI after load...")
            self._refresh_ui_after_load()

            # Emit PROJECT_LOADED event so panels can update themselves
            logger.info("Emitting PROJECT_LOADED event...")
            from core.state_manager import StateEvent, state_manager
            state_manager.emit(StateEvent.PROJECT_LOADED, source_id=None, data={}, component="ModularMainWindow")

            logger.info("Project load into UI complete")
        finally:
            if loading_dialog:
                loading_dialog.close()
    
    def _clear_ui_for_new_project(self):
        """Clear UI components for new project."""
        # Clear spatial view
        if hasattr(self, 'spatial_view') and self.spatial_view:
            self.spatial_view.sources = {}
            if hasattr(self.spatial_view, 'update_display'):
                self.spatial_view.update_display()
        
        # Refresh panels if they exist
        self._refresh_ui_after_load()
    
    def _refresh_ui_after_load(self):
        """Refresh UI components after project load."""
        logger.info("Refreshing UI components...")

        # Ensure Audio Sources panel exists and is visible for loaded projects
        if hasattr(self, 'audio_engine') and len(self.audio_engine.sources) > 0:
            logger.info(f"Project has {len(self.audio_engine.sources)} sources - ensuring Audio Sources panel is visible")
            if not hasattr(self, 'audio_sources_panel') or self.audio_sources_panel is None:
                logger.info("Audio Sources panel doesn't exist - creating it")
                self.panel_manager.show_panel("audio_sources")

        # Update audio sources panel
        if hasattr(self, 'audio_sources_panel') and self.audio_sources_panel is not None:
            logger.info("Refreshing Audio Sources panel")
            if hasattr(self.audio_sources_panel, 'refresh_source_list'):
                self.audio_sources_panel.refresh_source_list()
                logger.info("Audio Sources panel refreshed")
            else:
                logger.warning("Audio Sources panel has no refresh_source_list method")
        else:
            logger.warning("Audio Sources panel not available for refresh")

        # Update audio processing panel
        if hasattr(self, 'audio_processing_panel') and self.audio_processing_panel is not None:
            if hasattr(self.audio_processing_panel, 'refresh_sources'):
                self.audio_processing_panel.refresh_sources()

        # Update spatial positioning panel
        if hasattr(self, 'source_properties_panel') and self.source_properties_panel is not None:
            if hasattr(self.source_properties_panel, 'refresh_source_list'):
                self.source_properties_panel.refresh_source_list()

        # Update spatial view
        if hasattr(self, 'spatial_view') and self.spatial_view:
            # Update spatial view sources using proper add_source method
            if hasattr(self, 'audio_engine') and self.audio_engine.sources:
                for source_id, source in self.audio_engine.sources.items():
                    # Get source name from audio_bridge.loaded_samples
                    name = source_id  # Default to UUID
                    if hasattr(self, 'audio_bridge') and hasattr(self.audio_bridge, 'loaded_samples'):
                        if source_id in self.audio_bridge.loaded_samples:
                            sample_info = self.audio_bridge.loaded_samples[source_id]
                            if isinstance(sample_info, dict) and 'name' in sample_info:
                                name = sample_info['name']

                    # Use add_source if source doesn't exist, or update position if it does
                    if source_id not in self.spatial_view.sources:
                        self.spatial_view.add_source(source_id, name, source.position)
                    else:
                        # Update existing source's position properly
                        self.spatial_view.sources[source_id]['position'] = source.position
                        self.spatial_view.draw_source(source_id)
            if hasattr(self.spatial_view, 'update_display'):
                self.spatial_view.update_display()
    
    def _update_window_title(self):
        """Update window title with project name."""
        project_name = self.project_manager.get_project_name()
        modified_indicator = " •" if self.project_manager.is_project_modified() else ""
        self.root.title(f"SHAC Studio - {project_name}{modified_indicator}")
    
    def _update_recent_projects_menu(self):
        """Update the recent projects menu."""
        # Clear existing menu items
        self.recent_menu.delete(0, "end")
        
        # Get recent projects
        recent_projects = self.project_manager.get_recent_projects()
        
        if not recent_projects:
            self.recent_menu.add_command(label="(No recent projects)", state="disabled")
        else:
            for i, project_path in enumerate(recent_projects[:10]):  # Limit to 10
                project_name = Path(project_path).stem
                # Add accelerator for first 9 items
                accelerator = f"Ctrl+{i+1}" if i < 9 else None
                self.recent_menu.add_command(
                    label=f"{project_name}",
                    command=lambda p=project_path: self._open_recent_project(p),
                    accelerator=accelerator
                )
            
            self.recent_menu.add_separator()
            self.recent_menu.add_command(label="Clear Recent Projects", command=self._clear_recent_projects)
    
    def _open_recent_project(self, filepath: str):
        """Open a recent project."""
        if Path(filepath).exists():
            # Check for unsaved changes first
            if self.project_manager.is_project_modified():
                response = messagebox.askyesnocancel(
                    "Unsaved Changes",
                    "Save changes to current project before opening another?",
                    parent=self.root
                )
                if response is None:  # Cancel
                    return
                elif response:  # Yes, save
                    if not self.save_project():
                        return  # Save failed or was cancelled
            
            # Load the project
            if self.project_manager.load_project(filepath):
                self._load_project_into_ui()
                self._update_window_title()
                self._update_recent_projects_menu()  # Refresh menu order
            else:
                error_dialogs.show_error("Load Error", f"Failed to load project: {Path(filepath).name}\n\nThe file may be corrupted or in an incompatible format.", parent=self.root)
        else:
            messagebox.showerror("File Not Found", f"Project file no longer exists: {filepath}", parent=self.root)
            # Remove from recent projects
            if hasattr(self.project_manager, 'recent_projects'):
                self.project_manager.recent_projects = [p for p in self.project_manager.recent_projects if p != filepath]
                self.project_manager.save_recent_projects()
                self._update_recent_projects_menu()
    
    def _clear_recent_projects(self):
        """Clear the recent projects list."""
        if messagebox.askyesno("Clear Recent Projects", "Clear all recent projects from the menu?", parent=self.root):
            self.project_manager.recent_projects = []
            self.project_manager.save_recent_projects()
            self._update_recent_projects_menu()
        
    def export_shac(self):
        """Export to SHAC format."""
        # Ensure audio sources panel exists
        if not self.is_panel_created("audio_sources"):
            self.create_panel("audio_sources")
            
        dialog = ExportDialog(self.root, self.audio_sources_panel, self.audio_bridge)
        
    def show_help(self):
        """Show help documentation in a proper dialog."""
        # Create custom help dialog
        help_window = tk.Toplevel(self.root)
        help_window.title("SHAC Studio - Help")
        help_window.geometry("700x600")
        help_window.config(bg=theme.BACKGROUND_PRIMARY)

        # Create text widget with scrollbar
        text_frame = tk.Frame(help_window, bg=theme.BACKGROUND_PRIMARY)
        text_frame.pack(fill='both', expand=True, padx=20, pady=20)

        scrollbar = tk.Scrollbar(text_frame)
        scrollbar.pack(side='right', fill='y')

        text_widget = tk.Text(
            text_frame,
            wrap='word',
            font=('Monaco', 11),
            bg=theme.BACKGROUND_SECONDARY,
            fg=theme.TEXT_PRIMARY,
            padx=15,
            pady=15,
            yscrollcommand=scrollbar.set,
            relief='flat',
            borderwidth=0
        )
        text_widget.pack(side='left', fill='both', expand=True)
        scrollbar.config(command=text_widget.yview)

        help_text = """Getting Started with SHAC Studio

1. LOAD AUDIO FILES
   • View → Audio Sources to open the sources panel
   • Click "Add Audio Files" to import WAV, MP3, FLAC, or OGG files
   • Sources appear in the list with individual controls

2. POSITION SOURCES IN 3D SPACE
   • View → Spatial Player to open the spatial audio panel
   • Drag sources in the 3D view to position them
   • Use X, Y, Z sliders for precise positioning

3. EXPERIENCE SPATIAL AUDIO
   • Click Play to hear your spatial mix
   • Use WASD keys to walk around inside the music
   • Q/E keys move up and down
   • Solo/Mute individual sources for isolation

4. EXPORT INTERACTIVE AUDIO
   • File → Export to SHAC to create shareable files
   • Choose quality level (1st-7th order ambisonic)
   • Share .shac files that play in web browsers

KEYBOARD SHORTCUTS
   • Space: Play/Pause
   • R: Stop playback
   • W/A/S/D: Move forward/left/back/right
   • Q/E: Move up/down
   • Ctrl+Z: Undo last action
   • Ctrl+Shift+Z: Redo last undone action
   • Ctrl+N: New project
   • Ctrl+O: Open project
   • Ctrl+S: Save project
   • Ctrl+E: Export to SHAC
   • F1: Show this help

PANEL LAYOUTS
   • View → Layout → Default Layout: Standard workflow
   • View → Layout → Compact Layout: Minimal panels
   • View → Layout → Wide Layout: Spread across large screen

Press F1 anytime to show this help."""

        text_widget.insert('1.0', help_text)
        text_widget.config(state='disabled')  # Make read-only

        # Close button
        close_button = tk.Button(
            help_window,
            text="Close",
            command=help_window.destroy,
            bg=theme.ACCENT_PRIMARY,
            fg='#FFFFFF',
            font=('Helvetica', 12, 'bold'),
            padx=30,
            pady=10,
            relief='flat',
            cursor='hand2'
        )
        close_button.pack(pady=(0, 20))

        # Center window
        help_window.update_idletasks()
        x = (help_window.winfo_screenwidth() // 2) - (700 // 2)
        y = (help_window.winfo_screenheight() // 2) - (600 // 2)
        help_window.geometry(f"700x600+{x}+{y}")

        # Make modal
        help_window.transient(self.root)
        help_window.grab_set()
        help_window.focus_set()
        
    def show_about(self):
        """Show about dialog."""
        about_text = """SHAC Studio v1.0

Revolutionary Spatial Audio Creation Software

Create interactive audio experiences where listeners 
can walk around inside the music using WASD controls.

Patent-pending SHAC format technology
Co-invented by Clarke Czyz and Claude

🎵 The spatial audio revolution starts here 🎵"""
        
        messagebox.showinfo("About SHAC Studio", about_text, parent=self.root)
    
        
    def update_duration(self):
        """Update transport duration based on loaded sources."""
        duration = self.audio_engine.get_duration()
        if self.transport_panel:
            self.transport_panel.set_duration(duration)
        
    def on_playback_position_update(self, position: float):
        """Handle position updates from audio engine and distribute to all panels."""
        # Update transport panel if it exists
        if self.transport_panel and self.transport_panel.total_time > 0:
            percentage = (position / self.transport_panel.total_time) * 100
            self.transport_panel.timeline_var.set(percentage)
            self.transport_panel.current_time = position
            self.transport_panel.update_time_display()
            
        # Audio processing panel doesn't need playback position updates
        
    def start_autosave_timer(self):
        """Start periodic autosave timer."""
        # Perform autosave
        if hasattr(self, 'project_manager'):
            self._collect_project_data()
            self.project_manager.autosave_project()

        # Schedule next autosave (every 60 seconds, checks internally if enough time passed)
        if hasattr(self, 'root'):
            self.root.after(60000, self.start_autosave_timer)  # 60 seconds

    def check_autosave_recovery(self):
        """Check for autosave file on startup and offer to recover."""
        autosave_path = self.project_manager.check_for_autosave_recovery()

        if autosave_path:
            response = messagebox.askyesno(
                "Recover Project",
                "SHAC Studio found an autosaved project from a previous session.\n\n"
                "Would you like to recover it?",
                parent=self.root
            )

            if response:
                # Load the autosave file
                success = self.project_manager.load_project(autosave_path)
                if success:
                    self._load_project_into_ui()
                    error_dialogs.show_success(
                        "Project Recovered",
                        "Your project has been recovered from autosave.\n\n"
                        "Remember to save it with a proper name!",
                        parent=self.root
                    )
                else:
                    error_dialogs.show_error(
                        "Recovery Failed",
                        "Failed to recover the autosaved project.\n\nThe autosave file may be corrupted.",
                        parent=self.root
                    )
            else:
                # User declined - delete the autosave
                self.project_manager.delete_autosave()

    def run(self):
        """Start the application."""
        # Clean up on exit
        self.root.protocol("WM_DELETE_WINDOW", self.on_close)

        # Start autosave timer
        self.start_autosave_timer()

        # Check for autosave recovery on startup
        self.check_autosave_recovery()

        self.root.mainloop()

    def on_close(self):
        """Handle window close."""
        # Check for unsaved changes
        if self.project_manager.is_project_modified():
            response = messagebox.askyesnocancel(
                "Unsaved Changes",
                "Save changes before closing?",
                parent=self.root
            )
            if response is None:  # Cancel
                return
            elif response:  # Yes, save
                if not self.save_project():
                    return  # Save failed or was cancelled
        
        # Save panel states
        self.panel_manager.save_panel_states()
        
        # Stop audio engine
        if hasattr(self, 'audio_engine'):
            self.audio_engine.stop()
        
        # Close all panels
        for panel in self.panel_manager.panels.values():
            panel.hide()
            
        self.root.destroy()