"""
Spatial View - 3D positioning interface

The heart of SHAC Studio. This view shows audio sources in 3D space
and allows drag-and-drop positioning for creating spatial audio.
"""

import logging
import tkinter as tk
from tkinter import ttk
import math

# Set up logger
logger = logging.getLogger(__name__)


class SpatialView:
    """3D spatial positioning interface."""
    
    def __init__(self, parent):
        """Initialize the spatial view."""
        self.parent = parent
        self.sources = {}  # Dict of source_id: position
        self.selected_source = None
        self.drag_data = None
        self.source_panel = None  # Will be set by main window
        self.audio_engine = None  # Will be set by main window
        
        # Create main frame
        self.frame = ttk.Frame(parent)
        self.setup_ui()
        
    def setup_ui(self):
        """Set up the spatial view UI."""
        # Title at top
        top_frame = ttk.Frame(self.frame)
        top_frame.pack(fill='x', padx=5, pady=5)
        
        title = ttk.Label(top_frame, text="Spatial View", font=('TkDefaultFont', 12, 'bold'))
        title.pack(side='left')
        
        # Canvas for 3D visualization
        self.canvas = tk.Canvas(
            self.frame, 
            bg='#2b2b2b',
            highlightthickness=1,
            highlightbackground='#555'
        )
        self.canvas.pack(fill='both', expand=True, padx=5, pady=5)
        
        # Initialize position variables (now handled by main window)
        self.x_var = tk.DoubleVar(value=0.0)
        self.y_var = tk.DoubleVar(value=0.0)
        self.z_var = tk.DoubleVar(value=0.0)
        
        # Initialize listener position variables
        self.listener_x = tk.DoubleVar(value=0.0)
        self.listener_y = tk.DoubleVar(value=0.0) 
        self.listener_z = tk.DoubleVar(value=0.0)
        
        # Initialize volume variable
        self.volume_var = tk.DoubleVar(value=1.0)
        
        # Bind canvas events
        self.canvas.bind('<Configure>', self.on_canvas_resize)
        self.canvas.bind('<Button-1>', self.on_canvas_click)
        self.canvas.bind('<B1-Motion>', self.on_canvas_drag)
        self.canvas.bind('<ButtonRelease-1>', self.on_canvas_release)
        
        # Draw initial grid
        self.canvas.after(100, self.draw_grid)
        
    def draw_grid(self):
        """Draw reference grid on canvas."""
        self.canvas.delete('grid')
        
        width = self.canvas.winfo_width()
        height = self.canvas.winfo_height()
        
        if width < 10 or height < 10:
            return
            
        # Center point
        cx = width // 2
        cy = height // 2
        
        # Grid spacing
        spacing = 50
        
        # Draw grid lines
        for i in range(0, max(width, height), spacing):
            # Vertical lines
            x = cx + i
            if x < width:
                self.canvas.create_line(x, 0, x, height, fill='#444', tags='grid')
            x = cx - i
            if x > 0:
                self.canvas.create_line(x, 0, x, height, fill='#444', tags='grid')
                
            # Horizontal lines
            y = cy + i
            if y < height:
                self.canvas.create_line(0, y, width, y, fill='#444', tags='grid')
            y = cy - i
            if y > 0:
                self.canvas.create_line(0, y, width, y, fill='#444', tags='grid')
                
        # Draw axes
        self.canvas.create_line(cx, 0, cx, height, fill='#666', width=2, tags='grid')
        self.canvas.create_line(0, cy, width, cy, fill='#666', width=2, tags='grid')
        
        # Draw listener at spawn position
        listener_pos = self.get_listener_position()
        listener_canvas_x, listener_canvas_y = self.world_to_canvas(*listener_pos)
        
        self.canvas.create_oval(
            listener_canvas_x - 10, listener_canvas_y - 10, 
            listener_canvas_x + 10, listener_canvas_y + 10,
            fill='#4a9eff', outline='#fff', width=2, tags='grid'
        )
        self.canvas.create_text(
            listener_canvas_x, listener_canvas_y - 20, 
            text="Listener Spawn", fill='#ccc', tags='grid'
        )
        
    def add_source(self, source_id, name, position=(0, 0, 0)):
        """Add a source to the spatial view."""
        self.sources[source_id] = {
            'name': name,
            'position': position,
            'volume': 1.0,
            'canvas_id': None
        }
        self.draw_source(source_id)
        
        # Update audio engine position if available
        if self.audio_engine:
            self.audio_engine.set_source_position(source_id, position)
    
    def rename_source(self, source_id, new_name):
        """Rename a source in the spatial view."""
        if source_id in self.sources:
            old_name = self.sources[source_id]['name']
            self.sources[source_id]['name'] = new_name
            # Redraw to update the label
            self.draw_source(source_id)
            logger.debug(f"Spatial view: renamed source '{old_name}' to '{new_name}'")
        
    def draw_source(self, source_id):
        """Draw a source on the canvas."""
        if source_id not in self.sources:
            return
            
        source = self.sources[source_id]
        x, y, z = source['position']
        
        # Convert 3D position to 2D canvas coordinates
        canvas_x, canvas_y = self.world_to_canvas(x, y, z)
        
        # Remove old drawing if exists (both circle and text)
        self.canvas.delete(f'source_{source_id}')
            
        # Draw source circle
        radius = 15
        color = '#ff6b6b' if source_id == self.selected_source else '#ffa726'
        
        canvas_id = self.canvas.create_oval(
            canvas_x - radius, canvas_y - radius,
            canvas_x + radius, canvas_y + radius,
            fill=color, outline='#fff', width=2,
            tags=f'source_{source_id}'
        )
        
        # Draw elevation indicator (dotted line)
        if z != 0:
            # Scale factor for elevation visualization
            z_scale = 20  # pixels per meter of elevation
            line_length = abs(z) * z_scale
            
            if z > 0:
                # Above ground - dotted line going down from source
                line_end_y = canvas_y + radius + line_length
                self.canvas.create_line(
                    canvas_x, canvas_y + radius,
                    canvas_x, line_end_y,
                    fill='#aaa', stipple='gray50', width=2,
                    tags=f'source_{source_id}'
                )
                # Small ground indicator
                self.canvas.create_oval(
                    canvas_x - 3, line_end_y - 3,
                    canvas_x + 3, line_end_y + 3,
                    fill='#666', outline='#aaa',
                    tags=f'source_{source_id}'
                )
            else:
                # Below ground - dotted line going up from source
                line_end_y = canvas_y - radius - line_length
                self.canvas.create_line(
                    canvas_x, canvas_y - radius,
                    canvas_x, line_end_y,
                    fill='#aaa', stipple='gray50', width=2,
                    tags=f'source_{source_id}'
                )
                # Small ground indicator
                self.canvas.create_oval(
                    canvas_x - 3, line_end_y - 3,
                    canvas_x + 3, line_end_y + 3,
                    fill='#666', outline='#aaa',
                    tags=f'source_{source_id}'
                )

        # Draw label
        self.canvas.create_text(
            canvas_x, canvas_y - radius - 10,
            text=source['name'], fill='#fff',
            tags=f'source_{source_id}'
        )
        
        source['canvas_id'] = canvas_id
        
    def remove_source(self, source_id):
        """Remove a source from the spatial view."""
        if source_id in self.sources:
            source = self.sources[source_id]
            # Remove from canvas
            self.canvas.delete(f'source_{source_id}')
            # Remove from sources
            del self.sources[source_id]
            # Clear selection if this was selected
            if self.selected_source == source_id:
                self.selected_source = None
        
    def world_to_canvas(self, x, y, z):
        """Convert world coordinates to canvas coordinates."""
        width = self.canvas.winfo_width()
        height = self.canvas.winfo_height()
        
        if width <= 1 or height <= 1:
            return 100, 100  # Fallback for uninitialized canvas
        
        cx = width // 2
        cy = height // 2
        
        # Scale factor (pixels per meter)
        scale = 30
        
        # Top view by default
        canvas_x = cx + x * scale
        canvas_y = cy - y * scale  # Invert Y for standard coordinate system
        
        return int(canvas_x), int(canvas_y)
        
    def canvas_to_world(self, canvas_x, canvas_y):
        """Convert canvas coordinates to world coordinates."""
        width = self.canvas.winfo_width()
        height = self.canvas.winfo_height()
        
        if width <= 1 or height <= 1:
            return 0, 0, 0  # Fallback
        
        cx = width // 2
        cy = height // 2
        
        scale = 30
        
        x = (canvas_x - cx) / scale
        y = (cy - canvas_y) / scale  # Invert Y
        z = self.z_var.get() if self.selected_source else 0
        
        # Clamp to reasonable range
        x = max(-10, min(10, x))
        y = max(-10, min(10, y))
        
        return x, y, z
        
    def on_canvas_click(self, event):
        """Handle canvas click."""
        # Find if we clicked on a source
        clicked_items = self.canvas.find_overlapping(event.x-5, event.y-5, event.x+5, event.y+5)
        
        # Look for source items
        for item in clicked_items:
            tags = self.canvas.gettags(item)
            for tag in tags:
                if tag.startswith('source_'):
                    source_id = tag.replace('source_', '')
                    if source_id in self.sources:
                        logger.debug(f"Clicked on source: {source_id}")
                        self.select_source(source_id)
                        self.drag_data = {
                            'x': event.x, 
                            'y': event.y, 
                            'source': source_id,
                            'start_pos': self.sources[source_id]['position']
                        }
                        # Change cursor to indicate dragging
                        self.canvas.configure(cursor="hand2")
                        return
        
        # If no source clicked, clear selection
        self.drag_data = None
        self.selected_source = None
                
    def on_canvas_drag(self, event):
        """Handle canvas drag."""
        if self.drag_data and 'source' in self.drag_data:
            source_id = self.drag_data['source']
            
            if source_id in self.sources:
                # Convert current mouse position to world coordinates
                x, y, z = self.canvas_to_world(event.x, event.y)
                z = self.z_var.get()  # Keep Z from slider
                
                logger.debug(f"Dragging {source_id} to ({x:.1f}, {y:.1f}, {z:.1f})")
                
                # Update source position
                self.sources[source_id]['position'] = (x, y, z)
                
                # Redraw the source at new position
                self.draw_source(source_id)
                
                # Update controls to reflect new position
                self.x_var.set(x)
                self.y_var.set(y)
                self.z_var.set(z)
                
                # Notify positioning panel through parent window
                if hasattr(self.parent, 'source_properties_panel') and self.parent.source_properties_panel:
                    panel = self.parent.source_properties_panel
                    if hasattr(panel, '_source_positions'):
                        panel._source_positions[source_id] = (x, y, z)
                        if source_id == panel.current_source_id:
                            panel._update_position_display((x, y, z))
                
                # Update audio engine position if available
                if self.audio_engine:
                    self.audio_engine.set_source_position(source_id, (x, y, z))
                    
                # Update spatial player
                if hasattr(self.parent, 'spatial_player_panel') and self.parent.spatial_player_panel:
                    self.parent.spatial_player_panel.update_source_position(source_id, (x, y, z))
                    if hasattr(self.parent.spatial_player_panel, 'request_render'):
                        self.parent.spatial_player_panel.request_render()
            
    def on_canvas_release(self, event):
        """Handle canvas release."""
        if self.drag_data:
            logger.debug(f"Released source: {self.drag_data.get('source', 'unknown')}")
        self.drag_data = None
        self.canvas.configure(cursor="")
        
    def on_canvas_resize(self, event):
        """Handle canvas resize."""
        self.draw_grid()
        # Redraw all sources
        for source_id in self.sources:
            self.draw_source(source_id)
            
    def select_source(self, source_id):
        """Select a source."""
        self.selected_source = source_id
        
        if source_id in self.sources:
            x, y, z = self.sources[source_id]['position']
            self.x_var.set(x)
            self.y_var.set(y)
            self.z_var.set(z)
            
            # Update volume slider to show current source volume
            volume = self.sources[source_id].get('volume', 1.0)
            self.volume_var.set(volume)
            
        # Redraw all sources to update colors
        for sid in self.sources:
            self.draw_source(sid)
            
    def on_position_change(self, value):
        """Handle position slider change."""
        if self.selected_source and self.selected_source in self.sources:
            x = self.x_var.get()
            y = self.y_var.get()
            z = self.z_var.get()
            
            logger.debug(f"Position changed {self.selected_source} to ({x:.1f}, {y:.1f}, {z:.1f})")
            
            # Update source position
            self.sources[self.selected_source]['position'] = (x, y, z)
            self.draw_source(self.selected_source)
            
            # Notify positioning panel through parent window
            if hasattr(self.parent, 'source_properties_panel') and self.parent.source_properties_panel:
                panel = self.parent.source_properties_panel
                if hasattr(panel, '_source_positions'):
                    panel._source_positions[self.selected_source] = (x, y, z)
                    if self.selected_source == panel.current_source_id:
                        panel._update_position_display((x, y, z))
            
            # Update audio engine position if available
            if self.audio_engine:
                self.audio_engine.set_source_position(self.selected_source, (x, y, z))
                
            # Update spatial player
            if hasattr(self.parent, 'spatial_player_panel') and self.parent.spatial_player_panel:
                self.parent.spatial_player_panel.update_source_position(self.selected_source, (x, y, z))
                self.parent.spatial_player_panel.request_render()
                
    def on_entry_change(self, event):
        """Handle coordinate entry field changes."""
        if self.selected_source and self.selected_source in self.sources:
            try:
                x = float(self.x_var.get())
                y = float(self.y_var.get()) 
                z = float(self.z_var.get())
                
                # Clamp values to reasonable range
                x = max(-20, min(20, x))
                y = max(-20, min(20, y))
                z = max(-20, min(20, z))
                
                logger.debug(f"Entry changed {self.selected_source} to ({x:.1f}, {y:.1f}, {z:.1f})")
                
                # Update variables (this will trigger slider updates)
                self.x_var.set(x)
                self.y_var.set(y)
                self.z_var.set(z)
                
                # Update source position
                self.sources[self.selected_source]['position'] = (x, y, z)
                self.draw_source(self.selected_source)
                
                # Update source panel if connected
                if self.source_panel:
                    self.source_panel.update_source_position(self.selected_source, (x, y, z))
                    
                # Update audio engine position if available
                if self.audio_engine:
                    self.audio_engine.set_source_position(self.selected_source, (x, y, z))
                    
            except ValueError:
                logger.warning("Invalid coordinate value entered")
                # Reset to current position
                if self.selected_source in self.sources:
                    pos = self.sources[self.selected_source]['position']
                    self.x_var.set(pos[0])
                    self.y_var.set(pos[1])
                    self.z_var.set(pos[2])
                    
    def set_position(self, x, y, z):
        """Set position using preset buttons."""
        if self.selected_source and self.selected_source in self.sources:
            logger.debug(f"Preset position {self.selected_source} to ({x}, {y}, {z})")
            
            # Update entry fields and sliders
            self.x_var.set(x)
            self.y_var.set(y)
            self.z_var.set(z)
            
            # Update source position
            self.sources[self.selected_source]['position'] = (x, y, z)
            self.draw_source(self.selected_source)
            
            # Notify positioning panel through parent window
            if hasattr(self.parent, 'source_properties_panel') and self.parent.source_properties_panel:
                panel = self.parent.source_properties_panel
                if hasattr(panel, '_source_positions'):
                    panel._source_positions[self.selected_source] = (x, y, z)
                    if self.selected_source == panel.current_source_id:
                        panel._update_position_display((x, y, z))
            
            # Update audio engine position if available
            if self.audio_engine:
                self.audio_engine.set_source_position(self.selected_source, (x, y, z))
                
            # Update spatial player
            if hasattr(self.parent, 'spatial_player_panel') and self.parent.spatial_player_panel:
                self.parent.spatial_player_panel.update_source_position(self.selected_source, (x, y, z))
                self.parent.spatial_player_panel.request_render()
        else:
            logger.warning("No source selected for preset position")
            
    def set_listener_position(self, x, y, z):
        """Set listener spawn position."""
        logger.info(f"Setting listener spawn position to ({x}, {y}, {z})")
        self.listener_x.set(x)
        self.listener_y.set(y)
        self.listener_z.set(z)
        
        # Redraw the visual representation
        self.draw_grid()  # This will show the listener at the new position
        
    def get_listener_position(self):
        """Get current listener position."""
        return (self.listener_x.get(), self.listener_y.get(), self.listener_z.get())
        
    def on_volume_change(self, value):
        """Handle source volume change."""
        if self.selected_source and self.selected_source in self.sources:
            volume = self.volume_var.get()
            logger.debug(f"Volume changed for {self.selected_source} to {volume:.2f}")
            
            # Update source volume in local storage
            self.sources[self.selected_source]['volume'] = volume
            
            # Update audio engine if available
            if self.audio_engine:
                self.audio_engine.set_source_volume(self.selected_source, volume)
                
    def update_source_position(self, source_id, position):
        """Universal source position update method - the Lamborghini of positioning."""
        if source_id in self.sources:
            # Update local position
            self.sources[source_id]['position'] = position
            self.draw_source(source_id)
            
            # Update audio engine if available
            if self.audio_engine:
                self.audio_engine.set_source_position(source_id, position)
                
            # Update spatial player if available and ready
            if hasattr(self.parent, 'spatial_player_panel') and self.parent.spatial_player_panel:
                self.parent.spatial_player_panel.update_source_position(source_id, position)
                if hasattr(self.parent.spatial_player_panel, 'request_render'):
                    self.parent.spatial_player_panel.request_render()
                
            logger.debug(f"Position updated: {source_id} → {position}")
        else:
            logger.warning(f"Source {source_id} not found in spatial view")