"""
Export Dialog - SHAC export settings

Handles exporting the spatial audio project to .shac format
with quality options and file settings.
"""

import logging
import tkinter as tk
from tkinter import ttk, filedialog, messagebox
from pathlib import Path
from . import theme
from .icons import get_icon
from .theme_helpers import create_themed_button, create_themed_label, create_themed_frame

# Set up logger
logger = logging.getLogger(__name__)


class ExportDialog:
    """Dialog for exporting to SHAC format."""
    
    def __init__(self, parent, source_panel, audio_bridge):
        """Initialize export dialog."""
        self.parent = parent
        self.source_panel = source_panel
        self.audio_bridge = audio_bridge
        self.result = None
        
        # Create dialog window with professional theme
        self.dialog = tk.Toplevel(parent)
        self.dialog.title("Export to SHAC")
        self.dialog.geometry("480x750")  # Comfortably shows all export sections
        self.dialog.minsize(450, 650)  # Minimum size to keep UI usable
        self.dialog.transient(parent)
        self.dialog.grab_set()

        # Apply theme
        self.dialog.config(bg=theme.BACKGROUND_PRIMARY)

        # Configure ttk theme for dialog
        style = ttk.Style()
        theme.Theme.configure_ttk_style(style)
        
        self.setup_ui()
        
        # Center the dialog
        self.dialog.update_idletasks()
        x = (self.dialog.winfo_screenwidth() // 2) - (self.dialog.winfo_width() // 2)
        y = (self.dialog.winfo_screenheight() // 2) - (self.dialog.winfo_height() // 2)
        self.dialog.geometry(f"+{x}+{y}")
        
    def populate_source_names(self):
        """Populate the source names section with current sources."""
        # Clear existing widgets
        for widget in self.source_names_scrollable_frame.winfo_children():
            widget.destroy()
        
        self.export_name_vars.clear()
        
        # Get current sources from the source panel
        if hasattr(self.source_panel, 'sources') and self.source_panel.sources:
            for i, source in enumerate(self.source_panel.sources):
                source_id = source['id']
                source_name = source['name']
                
                # Create frame for this source
                source_frame = ttk.Frame(self.source_names_scrollable_frame)
                source_frame.pack(fill='x', pady=2)
                
                # Source label (working name)
                ttk.Label(
                    source_frame,
                    text=f"{source_name}:",
                    width=25,
                    anchor='w'
                ).pack(side='left', padx=(0, 10))
                
                # "Export as:" label
                ttk.Label(
                    source_frame,
                    text="Export as:",
                    font=('TkDefaultFont', 8),
                    foreground='#666'
                ).pack(side='left', padx=(0, 5))
                
                # Export name entry (defaults to source name)
                export_name_var = tk.StringVar(value=source_name)
                export_entry = ttk.Entry(
                    source_frame,
                    textvariable=export_name_var,
                    width=20
                )
                export_entry.pack(side='left', padx=(0, 5))
                
                # Store the variable for later retrieval
                self.export_name_vars[source_id] = export_name_var
                
        else:
            # No sources available
            ttk.Label(
                self.source_names_scrollable_frame,
                text="No audio sources loaded.",
                foreground='#999'
            ).pack(pady=20)
            
        # Update scroll region
        self.source_names_scrollable_frame.update_idletasks()
        self.source_names_canvas.configure(scrollregion=self.source_names_canvas.bbox("all"))
        
    def setup_ui(self):
        """Set up the dialog UI."""
        # Create canvas and scrollbars for entire dialog
        canvas = tk.Canvas(self.dialog, highlightthickness=0)
        v_scrollbar = ttk.Scrollbar(self.dialog, orient="vertical", command=canvas.yview)
        h_scrollbar = ttk.Scrollbar(self.dialog, orient="horizontal", command=canvas.xview)

        # Main frame with padding inside canvas
        main_frame = ttk.Frame(canvas, padding="20")

        # Configure canvas
        main_frame.bind(
            "<Configure>",
            lambda e: canvas.configure(scrollregion=canvas.bbox("all"))
        )

        canvas.create_window((0, 0), window=main_frame, anchor="nw")
        canvas.configure(yscrollcommand=v_scrollbar.set, xscrollcommand=h_scrollbar.set)

        # Pack canvas and scrollbars
        canvas.grid(row=0, column=0, sticky="nsew")
        v_scrollbar.grid(row=0, column=1, sticky="ns")
        h_scrollbar.grid(row=1, column=0, sticky="ew")

        # Configure grid weights
        self.dialog.grid_rowconfigure(0, weight=1)
        self.dialog.grid_columnconfigure(0, weight=1)

        # Enable mousewheel scrolling
        def _on_mousewheel(event):
            canvas.yview_scroll(int(-1*(event.delta/120)), "units")
        def _on_shift_mousewheel(event):
            canvas.xview_scroll(int(-1*(event.delta/120)), "units")
        canvas.bind_all("<MouseWheel>", _on_mousewheel)
        canvas.bind_all("<Shift-MouseWheel>", _on_shift_mousewheel)
        
        # File location
        file_frame = ttk.LabelFrame(main_frame, text="Export Location", padding="10")
        file_frame.pack(fill='x', pady=(0, 10))

        self.file_path = tk.StringVar(value="untitled.shac")  # Default to SHAC format
        
        ttk.Label(file_frame, text="Filename:").grid(row=0, column=0, sticky='w', pady=5)
        
        path_frame = ttk.Frame(file_frame)
        path_frame.grid(row=1, column=0, sticky='ew', pady=5)
        file_frame.columnconfigure(0, weight=1)
        
        self.path_entry = ttk.Entry(path_frame, textvariable=self.file_path)
        self.path_entry.pack(side='left', fill='x', expand=True)
        
        ttk.Button(
            path_frame,
            text="Browse...",
            command=self.browse_file
        ).pack(side='left', padx=(5, 0))

        # Format is now always SHAC - no selection needed
        self.format_var = tk.StringVar(value="shac")

        # Quality settings
        quality_frame = ttk.LabelFrame(main_frame, text="Quality Settings", padding="10")
        quality_frame.pack(fill='x', pady=(0, 10))
        
        ttk.Label(quality_frame, text="Ambisonic Order:").grid(row=0, column=0, sticky='w', pady=5)
        
        self.order_var = tk.IntVar(value=3)
        order_frame = ttk.Frame(quality_frame)
        order_frame.grid(row=1, column=0, sticky='w', pady=5)
        
        orders = [
            (1, "1st Order (4 channels) - Smallest file"),
            (3, "3rd Order (16 channels) - Recommended"),
            (5, "5th Order (36 channels) - High quality"),
            (7, "7th Order (64 channels) - Maximum quality")
        ]
        
        for order, label in orders:
            ttk.Radiobutton(
                order_frame,
                text=label,
                variable=self.order_var,
                value=order
            ).pack(anchor='w', pady=2)
            
        # File size estimate - REMOVED (was wildly inaccurate)
        # self.size_label = ttk.Label(quality_frame, text="Estimated file size: ~10 MB")
        # self.size_label.grid(row=2, column=0, sticky='w', pady=(10, 5))
        
        # Source names section
        source_names_frame = ttk.LabelFrame(main_frame, text="Source Names (What Players See)", padding="10")
        source_names_frame.pack(fill='x', pady=(0, 10))
        
        # Explanation
        ttk.Label(
            source_names_frame,
            text="Customize how each source appears in the web player:",
            font=('TkDefaultFont', 9),
            foreground='#555'
        ).pack(anchor='w', pady=(0, 10))
        
        # Scrollable frame for source names
        self.source_names_canvas = tk.Canvas(source_names_frame, height=120)
        self.source_names_scrollbar = ttk.Scrollbar(source_names_frame, orient="vertical", command=self.source_names_canvas.yview)
        self.source_names_scrollable_frame = ttk.Frame(self.source_names_canvas)
        
        self.source_names_scrollable_frame.bind(
            "<Configure>",
            lambda e: self.source_names_canvas.configure(scrollregion=self.source_names_canvas.bbox("all"))
        )
        
        self.source_names_canvas.create_window((0, 0), window=self.source_names_scrollable_frame, anchor="nw")
        self.source_names_canvas.configure(yscrollcommand=self.source_names_scrollbar.set)
        
        self.source_names_canvas.pack(side="left", fill="both", expand=True)
        self.source_names_scrollbar.pack(side="right", fill="y")
        
        # Dictionary to hold export name variables
        self.export_name_vars = {}
        
        # Populate source names (will be called after dialog is created)
        self.populate_source_names()
        
        # Audio settings
        audio_frame = ttk.LabelFrame(main_frame, text="Audio Settings", padding="10")
        audio_frame.pack(fill='x', pady=(0, 10))
        
        # Normalize
        self.normalize_var = tk.BooleanVar(value=True)
        ttk.Checkbutton(
            audio_frame,
            text="Normalize audio levels (equalizes volume between sources)",
            variable=self.normalize_var
        ).pack(anchor='w', pady=5)
        
        # Distance-based gain compensation
        self.distance_gain_var = tk.BooleanVar(value=False)  # Default OFF
        distance_frame = ttk.Frame(audio_frame)
        distance_frame.pack(anchor='w', pady=5)
        
        ttk.Checkbutton(
            distance_frame,
            text="Apply distance-based gain compensation",
            variable=self.distance_gain_var
        ).pack(anchor='w')
        
        # Explanation label
        ttk.Label(
            distance_frame,
            text="⚠️  Adjusts volume based on source position - may dramatically alter levels",
            font=('TkDefaultFont', 8),
            foreground='#666'
        ).pack(anchor='w', padx=20)
        
        # Source length handling
        length_frame = ttk.LabelFrame(audio_frame, text="Source Length Handling", padding="5")
        length_frame.pack(fill='x', pady=(10, 5))
        
        self.length_handling_var = tk.StringVar(value="pad")
        
        ttk.Radiobutton(
            length_frame,
            text="Pad shorter sources with silence (recommended)",
            variable=self.length_handling_var,
            value="pad"
        ).pack(anchor='w', pady=2)
        
        ttk.Radiobutton(
            length_frame,
            text="Trim all sources to shortest length (smaller file size)",
            variable=self.length_handling_var,
            value="trim"
        ).pack(anchor='w', pady=2)
        
        ttk.Radiobutton(
            length_frame,
            text="Time-stretch to match longest (experimental - may degrade quality)",
            variable=self.length_handling_var,
            value="stretch"
        ).pack(anchor='w', pady=2)
        
        # Sample rate
        rate_label_frame = ttk.Frame(audio_frame)
        rate_label_frame.pack(anchor='w', pady=(10, 5))
        
        ttk.Label(rate_label_frame, text="Sample Rate:").pack(side='left')
        ttk.Label(
            rate_label_frame,
            text="(sources will be resampled if different)",
            font=('TkDefaultFont', 8),
            foreground='#666'
        ).pack(side='left', padx=(5, 0))

        # Determine default sample rate - use highest from loaded sources to avoid upsampling
        default_sample_rate = "44100"  # Fallback
        if hasattr(self.audio_bridge, 'loaded_samples') and self.audio_bridge.loaded_samples:
            sample_rates = []
            for sample_info in self.audio_bridge.loaded_samples.values():
                sr = sample_info.get('sample_rate', 44100)
                sample_rates.append(sr)
            if sample_rates:
                max_rate = max(sample_rates)
                # Round to nearest available option (44100, 48000, 96000)
                available_rates = [44100, 48000, 96000]
                # Pick the closest rate that's >= max_rate (avoid downsampling)
                best_rate = None
                for rate in available_rates:
                    if rate >= max_rate:
                        best_rate = rate
                        break
                if best_rate is None:
                    best_rate = available_rates[-1]  # Use highest if source is > 96kHz
                default_sample_rate = str(best_rate)
                logger.info(f"Auto-detected sample rate: {default_sample_rate} Hz (from source max: {max_rate} Hz)")

        self.sample_rate_var = tk.StringVar(value=default_sample_rate)

        rate_frame = ttk.Frame(audio_frame)
        rate_frame.pack(anchor='w')
        
        rates = [("44100", "44.1 kHz"), ("48000", "48 kHz"), ("96000", "96 kHz")]
        for rate, label in rates:
            ttk.Radiobutton(
                rate_frame,
                text=label,
                variable=self.sample_rate_var,
                value=rate
            ).pack(side='left', padx=5)
            
        # Progress frame (hidden initially)
        self.progress_frame = ttk.Frame(main_frame)
        
        self.progress_label = ttk.Label(self.progress_frame, text="Exporting...")
        self.progress_label.pack(pady=5)
        
        self.progress_var = tk.DoubleVar()
        self.progress_bar = ttk.Progressbar(
            self.progress_frame,
            variable=self.progress_var,
            maximum=100
        )
        self.progress_bar.pack(fill='x', pady=5)
        
        # Buttons - use fill='x' and anchor at bottom
        button_frame = ttk.Frame(main_frame)
        button_frame.pack(fill='x', side='bottom', pady=(20, 0))
        
        # Center the buttons
        ttk.Button(
            button_frame,
            text="Cancel",
            command=self.cancel
        ).pack(side='right', padx=5)
        
        self.export_button = ttk.Button(
            button_frame,
            text="Export",
            command=self.export
        )
        self.export_button.pack(side='right', padx=5)
        
        # Bind length handling change to show warning for stretch
        self.length_handling_var.trace('w', self.check_stretch_warning)

        # Bind format change to update file extension
        self.format_var.trace('w', self.update_file_extension)
        
    def browse_file(self):
        """Browse for export location."""
        filename = filedialog.asksaveasfilename(
            defaultextension=".shac",
            filetypes=[("SHAC files", "*.shac"), ("All files", "*.*")],
            initialfile=self.file_path.get()
        )
        if filename:
            self.file_path.set(filename)
            
    def update_file_extension(self, *args):
        """Ensure file has .shac extension."""
        current_path = self.file_path.get()

        # Remove old extension
        if current_path.endswith('.shac'):
            return  # Already has correct extension
        elif current_path.endswith('.zyz'):
            base_path = current_path[:-4]  # Remove old .zyz extension
        else:
            base_path = current_path

        # Always use .shac extension
        self.file_path.set(base_path + '.shac')

    def check_stretch_warning(self, *args):
        """Show warning if stretch option is selected."""
        if self.length_handling_var.get() == "stretch":
            response = tk.messagebox.askyesno(
                "Experimental Feature Warning",
                "Time-stretching audio sources can significantly degrade quality.\n\n" +
                "This feature is experimental and not recommended for regular use.\n\n" +
                "Are you sure you want to use time-stretching?",
                icon="warning"
            )
            if not response:
                self.length_handling_var.set("pad")  # Reset to default
        
    def export(self):
        """Start export process."""
        # Check if we have sources
        if not self.source_panel.sources:
            tk.messagebox.showerror("No Sources", "Please add audio sources before exporting.")
            return
        
        # Show progress
        self.export_button.config(state='disabled')
        self.progress_frame.pack(fill='x', pady=10)
        
        # Collect settings
        filename = self.file_path.get()
        order = self.order_var.get()
        normalize = self.normalize_var.get()
        distance_gain = self.distance_gain_var.get()
        length_handling = self.length_handling_var.get()
        sample_rate = int(self.sample_rate_var.get())
        
        # Validate and collect export names
        export_names = {}
        used_names = set()
        
        for source in self.source_panel.sources:
            source_id = source['id']
            # Get custom export name or fallback to source name
            if source_id in self.export_name_vars:
                export_name = self.export_name_vars[source_id].get().strip()
            else:
                export_name = source['name']
            
            # Validate export name
            if not export_name:
                tk.messagebox.showerror("Invalid Export Name", f"Export name for '{source['name']}' cannot be empty.")
                self.export_button.config(state='normal')
                self.progress_frame.pack_forget()
                return
                
            # Check for duplicate export names
            if export_name in used_names:
                tk.messagebox.showerror("Duplicate Export Names", f"Export name '{export_name}' is used by multiple sources. Each source must have a unique export name.")
                self.export_button.config(state='normal')
                self.progress_frame.pack_forget()
                return
                
            used_names.add(export_name)
            export_names[source_id] = export_name
        
        # Prepare sources for export with export names
        sources = []
        for source in self.source_panel.sources:
            # Get current position and volume from spatial_view (has real-time data)
            position = source['position']  # Default from audio_sources_panel
            volume = 1.0

            if hasattr(self.source_panel, 'spatial_view') and self.source_panel.spatial_view:
                if source['id'] in self.source_panel.spatial_view.sources:
                    spatial_source = self.source_panel.spatial_view.sources[source['id']]
                    if isinstance(spatial_source, dict):
                        # Get current position from spatial_view (updated in real-time)
                        position = spatial_source.get('position', position)
                        volume = spatial_source.get('volume', 1.0)

            sources.append({
                'id': source['id'],
                'position': position,
                'volume': volume,
                'export_name': export_names[source['id']]  # Add the custom export name
            })
        
        # Get listener spawn position from spatial view
        listener_position = (0, 0, 0)  # Default
        if hasattr(self.source_panel, 'spatial_view') and self.source_panel.spatial_view:
            listener_position = self.source_panel.spatial_view.get_listener_position()
        
        logger.info(f"Exporting {len(sources)} sources to {filename}")
        logger.debug(f"Listener spawn position: {listener_position}")
        logger.debug(f"Export names: {[s['export_name'] for s in sources]}")
        logger.debug(f"Export positions: {[(s['export_name'], s['position']) for s in sources]}")
        
        # Start export in separate thread to avoid blocking UI
        import threading
        export_thread = threading.Thread(
            target=self.perform_export,
            args=(sources, filename, order, sample_rate, normalize, distance_gain, length_handling, listener_position)
        )
        export_thread.start()
        
    def perform_export(self, sources, filename, order, sample_rate, normalize, distance_gain, length_handling, listener_position=(0, 0, 0)):
        """Perform the actual export in a separate thread."""
        try:
            # Update progress
            self.dialog.after(0, lambda: self.progress_var.set(10))
            self.dialog.after(0, lambda: self.progress_label.config(text="Preparing sources..."))
            
            # Normalize if requested
            if normalize:
                source_ids = [s['id'] for s in sources]
                self.audio_bridge.normalize_audio_levels(source_ids)
                
            self.dialog.after(0, lambda: self.progress_var.set(30))
            self.dialog.after(0, lambda: self.progress_label.config(text="Creating spatial mix..."))

            # Create progress callback
            num_sources = len(sources)
            def progress_callback(source_index, source_name, stage):
                """Update progress for each source."""
                # Progress from 30% to 95% based on source processing
                progress = 30 + int((source_index / num_sources) * 65)
                self.dialog.after(0, lambda: self.progress_var.set(progress))
                self.dialog.after(0, lambda: self.progress_label.config(text=f"{stage}: {source_name} ({source_index+1}/{num_sources})"))

            # SHAC format - multi-source layers
            success = self.audio_bridge.create_shac_composition(
                sources, filename, order, sample_rate, listener_position,
                distance_gain=distance_gain, length_handling=length_handling,
                progress_callback=progress_callback
            )

            self.dialog.after(0, lambda: self.progress_var.set(100))
            
            if success:
                self.dialog.after(0, lambda: self.progress_label.config(text="✅ Export complete!"))
                self.dialog.after(2000, self.dialog.destroy)
            else:
                self.dialog.after(0, lambda: self.progress_label.config(text="❌ Export failed!"))
                self.dialog.after(0, lambda: self.export_button.config(state='normal'))
                
        except Exception as e:
            logger.error(f"Export error: {e}", exc_info=True)
            self.dialog.after(0, lambda: self.progress_label.config(text="❌ Export failed!"))
            self.dialog.after(0, lambda: self.export_button.config(state='normal'))
            
    def cancel(self):
        """Cancel export."""
        self.result = None
        self.dialog.destroy()