"""
User-Friendly Error Dialog System

Provides themed, professional error messages to users while logging
technical details for debugging. Hides stack traces from users.
"""

import logging
import tkinter as tk
from tkinter import messagebox
import traceback
from . import theme

logger = logging.getLogger(__name__)


class ErrorDialog:
    """User-friendly error dialog with professional theming."""

    @staticmethod
    def show_error(title, message, technical_details=None, parent=None):
        """
        Show a user-friendly error dialog.

        Args:
            title: Short error title
            message: User-friendly message
            technical_details: Technical error info (logged but not shown)
            parent: Parent window
        """
        # Log technical details if provided
        if technical_details:
            logger.error(f"{title}: {message}")
            logger.error(f"Technical details: {technical_details}", exc_info=True)
        else:
            logger.error(f"{title}: {message}")

        # Show user-friendly message
        messagebox.showerror(title, message, parent=parent)

    @staticmethod
    def show_warning(title, message, parent=None):
        """
        Show a user-friendly warning dialog.

        Args:
            title: Short warning title
            message: User-friendly message
            parent: Parent window
        """
        logger.warning(f"{title}: {message}")
        messagebox.showwarning(title, message, parent=parent)

    @staticmethod
    def show_info(title, message, parent=None):
        """
        Show a user-friendly info dialog.

        Args:
            title: Short info title
            message: User-friendly message
            parent: Parent window
        """
        logger.info(f"{title}: {message}")
        messagebox.showinfo(title, message, parent=parent)

    @staticmethod
    def show_success(title, message, parent=None):
        """
        Show a success message dialog.

        Args:
            title: Short success title
            message: Success message
            parent: Parent window
        """
        logger.info(f"{title}: {message}")
        messagebox.showinfo(title, message, parent=parent)

    @staticmethod
    def handle_exception(operation_name, exception, parent=None):
        """
        Handle an exception with a user-friendly dialog.

        Args:
            operation_name: What was being attempted (e.g., "load audio file")
            exception: The exception object
            parent: Parent window
        """
        # Get the exception traceback
        tb = traceback.format_exc()

        # Log the full technical details
        logger.error(f"Error during {operation_name}: {str(exception)}")
        logger.error(f"Traceback:\n{tb}")

        # Show user-friendly message
        user_message = f"Failed to {operation_name}.\n\n{str(exception)}"
        messagebox.showerror(
            f"Error: {operation_name.capitalize()}",
            user_message,
            parent=parent
        )

    @staticmethod
    def ask_yes_no(title, message, parent=None):
        """
        Ask user a yes/no question.

        Args:
            title: Dialog title
            message: Question to ask
            parent: Parent window

        Returns:
            True if yes, False if no
        """
        return messagebox.askyesno(title, message, parent=parent)

    @staticmethod
    def ask_yes_no_cancel(title, message, parent=None):
        """
        Ask user a yes/no/cancel question.

        Args:
            title: Dialog title
            message: Question to ask
            parent: Parent window

        Returns:
            True if yes, False if no, None if cancel
        """
        return messagebox.askyesnocancel(title, message, parent=parent)


class LoadingDialog:
    """Themed loading/progress dialog."""

    def __init__(self, parent, title="Loading", message="Please wait..."):
        """
        Create a loading dialog.

        Args:
            parent: Parent window
            title: Dialog title
            message: Loading message
        """
        self.window = tk.Toplevel(parent)
        self.window.title(title)
        self.window.transient(parent)
        self.window.grab_set()

        # Apply theme
        self.window.config(bg=theme.BACKGROUND_SECONDARY)

        # Center on parent
        self.window.geometry("400x150")
        self._center_on_parent(parent)

        # Create content with theme
        content_frame = tk.Frame(
            self.window,
            bg=theme.BACKGROUND_SECONDARY,
            padx=30,
            pady=30
        )
        content_frame.pack(fill='both', expand=True)

        # Title label with theme
        self.title_label = tk.Label(
            content_frame,
            text=title,
            font=theme.FONT_TITLE,
            fg=theme.TEXT_ACCENT,
            bg=theme.BACKGROUND_SECONDARY
        )
        self.title_label.pack(pady=(0, 10))

        # Message label with theme
        self.message_label = tk.Label(
            content_frame,
            text=message,
            font=theme.FONT_NORMAL,
            fg=theme.TEXT_PRIMARY,
            bg=theme.BACKGROUND_SECONDARY,
            wraplength=340
        )
        self.message_label.pack(pady=(0, 20))

        # Progress bar with theme (indeterminate mode)
        from tkinter import ttk
        style = ttk.Style()
        style.configure(
            'Loading.Horizontal.TProgressbar',
            troughcolor=theme.SLIDER_TRACK,
            background=theme.ACCENT_PRIMARY,
            borderwidth=0,
            lightcolor=theme.ACCENT_PRIMARY,
            darkcolor=theme.ACCENT_PRIMARY
        )

        self.progress = ttk.Progressbar(
            content_frame,
            mode='indeterminate',
            style='Loading.Horizontal.TProgressbar',
            length=340
        )
        self.progress.pack(pady=(0, 10))
        self.progress.start(10)  # Animate

        # Update the window
        self.window.update()

    def _center_on_parent(self, parent):
        """Center dialog on parent window."""
        if parent:
            parent.update_idletasks()
            x = parent.winfo_x() + (parent.winfo_width() // 2) - 200
            y = parent.winfo_y() + (parent.winfo_height() // 2) - 75
            self.window.geometry(f"+{x}+{y}")

    def update_message(self, message):
        """Update the loading message."""
        self.message_label.config(text=message)
        self.window.update()

    def close(self):
        """Close the loading dialog."""
        self.progress.stop()
        self.window.grab_release()
        self.window.destroy()


# Convenience functions for quick access
def show_error(title, message, technical_details=None, parent=None):
    """Show user-friendly error."""
    ErrorDialog.show_error(title, message, technical_details, parent)


def show_warning(title, message, parent=None):
    """Show user-friendly warning."""
    ErrorDialog.show_warning(title, message, parent)


def show_info(title, message, parent=None):
    """Show user-friendly info."""
    ErrorDialog.show_info(title, message, parent)


def show_success(title, message, parent=None):
    """Show success message."""
    ErrorDialog.show_success(title, message, parent)


def handle_exception(operation_name, exception, parent=None):
    """Handle exception with user-friendly dialog."""
    ErrorDialog.handle_exception(operation_name, exception, parent)


def ask_yes_no(title, message, parent=None):
    """Ask yes/no question."""
    return ErrorDialog.ask_yes_no(title, message, parent)


def ask_yes_no_cancel(title, message, parent=None):
    """Ask yes/no/cancel question."""
    return ErrorDialog.ask_yes_no_cancel(title, message, parent)
