"""
Noise generation with spatial positioning for Atabey Symphony.

Provides various types of noise sources (white, pink, brown) with 3D positioning
capabilities for creating ambient textures and weather-inspired sound layers.
This module transforms Weather-Tune's noise.js into spatial audio components.
"""

import numpy as np
import logging
from typing import Optional, Dict, Any, Tuple
from abc import ABC, abstractmethod
try:
    import scipy.signal
    HAS_SCIPY = True
except ImportError:
    HAS_SCIPY = False


logger = logging.getLogger(__name__)


class NoiseGenerator(ABC):
    """Base class for different types of noise generators."""
    
    @abstractmethod
    def generate(self, duration: float, sample_rate: int) -> np.ndarray:
        """Generate noise signal."""
        pass


class WhiteNoiseGenerator(NoiseGenerator):
    """White noise generator - equal energy per frequency."""
    
    def generate(self, duration: float, sample_rate: int) -> np.ndarray:
        """Generate white noise with equal energy per frequency."""
        num_samples = int(duration * sample_rate)
        return np.random.uniform(-1, 1, num_samples)


class PinkNoiseGenerator(NoiseGenerator):
    """Pink noise generator - equal energy per octave.
    
    Uses Paul Kellett's refined method for pink noise generation.
    """
    
    def __init__(self):
        # State variables for pink noise filter
        self.b0 = 0.0
        self.b1 = 0.0
        self.b2 = 0.0
        self.b3 = 0.0
        self.b4 = 0.0
        self.b5 = 0.0
        self.b6 = 0.0
    
    def generate(self, duration: float, sample_rate: int) -> np.ndarray:
        """Generate pink noise using Paul Kellett's method."""
        num_samples = int(duration * sample_rate)
        output = np.zeros(num_samples)
        
        for i in range(num_samples):
            # Generate white noise
            white = np.random.uniform(-1, 1)
            
            # Apply pink filter
            self.b0 = 0.99886 * self.b0 + white * 0.0555179
            self.b1 = 0.99332 * self.b1 + white * 0.0750759
            self.b2 = 0.96900 * self.b2 + white * 0.1538520
            self.b3 = 0.86650 * self.b3 + white * 0.3104856
            self.b4 = 0.55000 * self.b4 + white * 0.5329522
            self.b5 = -0.7616 * self.b5 - white * 0.0168980
            
            # Combination
            pink = self.b0 + self.b1 + self.b2 + self.b3 + self.b4 + self.b5 + self.b6 + white * 0.5362
            self.b6 = white * 0.115926
            
            # Normalize to prevent clipping
            output[i] = pink * 0.11
        
        return output


class BrownNoiseGenerator(NoiseGenerator):
    """Brown noise generator - 6dB per octave rolloff."""
    
    def __init__(self):
        self.last_value = 0.0
    
    def generate(self, duration: float, sample_rate: int) -> np.ndarray:
        """Generate brown noise through integration of white noise."""
        num_samples = int(duration * sample_rate)
        output = np.zeros(num_samples)
        
        for i in range(num_samples):
            # Generate white noise
            white = np.random.uniform(-1, 1)
            
            # Integrate white noise
            self.last_value = (self.last_value + (0.02 * white)) / 1.02
            
            # Scale to compensate for low energy
            output[i] = self.last_value * 3.5
        
        return output


class SpatialNoise:
    """
    Creates various types of noise with 3D spatial positioning.
    Inspired by Weather-Tune's noise.js but enhanced for spatial audio.
    """
    
    def __init__(self):
        self.generators = {
            'white': WhiteNoiseGenerator(),
            'pink': PinkNoiseGenerator(),
            'brown': BrownNoiseGenerator()
        }
    
    def create_noise(self, options: Optional[Dict[str, Any]] = None) -> Dict[str, Any]:
        """
        Create a spatial noise source.
        
        Args:
            options: Configuration dictionary with:
                - type: 'white', 'pink', or 'brown' (default: 'white')
                - volume: Base volume 0-1 (default: 0.1)
                - duration: Duration in seconds (default: 2.0)
                - loop: Whether to loop (default: True)
                - filter_type: Biquad filter type (default: 'none')
                - filter_freq: Filter frequency (default: 1000)
                - filter_q: Filter Q value (default: 1.0)
                - position: Spatial position [x, y, z] (default: [0, 0, 0])
                - movement_pattern: Optional movement pattern
                - spatial_width: Spatial width factor (default: 1.0)
                
        Returns:
            Dictionary containing:
                - buffer: Audio buffer as numpy array
                - position: Spatial position
                - parameters: Audio parameters
                - shac_source: SHAC source configuration
        """
        if options is None:
            options = {}
        
        # Default parameters
        noise_type = options.get('type', 'white')
        volume = options.get('volume', 0.1)
        duration = options.get('duration', 2.0)
        loop = options.get('loop', True)
        filter_type = options.get('filter_type', 'none')
        filter_freq = options.get('filter_freq', 1000)
        filter_q = options.get('filter_q', 1.0)
        position = options.get('position', [0, 0, 0])
        movement_pattern = options.get('movement_pattern', None)
        spatial_width = options.get('spatial_width', 1.0)
        
        try:
            # Generate noise using appropriate generator
            generator = self.generators.get(noise_type, self.generators['white'])
            sample_rate = 44100  # Standard sample rate
            buffer = generator.generate(duration, sample_rate)
            
            # Apply volume
            buffer *= volume
            
            # Apply filter if requested
            if filter_type != 'none':
                buffer = self._apply_filter(buffer, filter_type, filter_freq, filter_q, sample_rate)
            
            # Create SHAC source configuration
            shac_source = {
                'type': 'noise',
                'noise_type': noise_type,
                'position': position,
                'volume': volume,
                'spatial_width': spatial_width,
                'loop': loop
            }
            
            # Only add movement pattern if it exists
            if movement_pattern is not None:
                shac_source['movement_pattern'] = movement_pattern
            
            return {
                'buffer': buffer,
                'position': position,
                'parameters': {
                    'type': noise_type,
                    'volume': volume,
                    'duration': duration,
                    'filter_type': filter_type,
                    'filter_freq': filter_freq,
                    'filter_q': filter_q,
                    'loop': loop
                },
                'shac_source': shac_source
            }
            
        except Exception as e:
            logger.error(f"Error creating spatial noise: {e}")
            return None
    
    def _apply_filter(self, signal: np.ndarray, filter_type: str, 
                     freq: float, q: float, sample_rate: int) -> np.ndarray:
        """Apply biquad filter to signal."""
        if not HAS_SCIPY:
            logger.warning("scipy not available, returning unfiltered signal")
            return signal
            
        try:
            nyquist = sample_rate / 2
            normalized_freq = freq / nyquist
            
            # Ensure frequency is within valid range
            normalized_freq = np.clip(normalized_freq, 0.01, 0.99)
            
            # Design filter based on type
            if filter_type == 'lowpass':
                b, a = scipy.signal.butter(2, normalized_freq, btype='low')
            elif filter_type == 'highpass':
                b, a = scipy.signal.butter(2, normalized_freq, btype='high')
            elif filter_type == 'bandpass':
                # For bandpass, we need a frequency range
                low_freq = normalized_freq * 0.8
                high_freq = normalized_freq * 1.2
                low_freq = np.clip(low_freq, 0.01, 0.99)
                high_freq = np.clip(high_freq, low_freq + 0.01, 0.99)
                b, a = scipy.signal.butter(2, [low_freq, high_freq], btype='band')
            elif filter_type == 'lowshelf' or filter_type == 'highshelf':
                # Approximate shelf filters with high/low pass
                if filter_type == 'lowshelf':
                    b, a = scipy.signal.butter(1, normalized_freq, btype='low')
                else:
                    b, a = scipy.signal.butter(1, normalized_freq, btype='high')
            else:
                return signal  # No filter applied
            
            # Apply filter
            filtered = scipy.signal.filtfilt(b, a, signal)
            return filtered
            
        except Exception as e:
            logger.error(f"Error applying filter: {e}")
            return signal
    
    def create_ambient_texture(self, weather_conditions: Dict[str, Any]) -> Dict[str, Any]:
        """
        Create ambient texture based on weather conditions.
        
        Args:
            weather_conditions: Weather data dictionary
            
        Returns:
            Dictionary with texture configuration
        """
        try:
            # Extract weather parameters
            temperature = weather_conditions.get('temperature', 20)
            humidity = weather_conditions.get('humidity', 50)
            wind_speed = weather_conditions.get('wind_speed', 0)
            weather_type = weather_conditions.get('type', 'clear')
            
            # Create texture based on weather
            texture_config = {
                'type': 'pink',  # Default to pink noise for natural feel
                'volume': 0.05,
                'duration': 3.0,
                'loop': True,
                'position': [0, 2, -1],  # Above and behind listener
                'spatial_width': 2.0
            }
            
            # Adjust based on weather type
            if weather_type == 'rain':
                texture_config.update({
                    'type': 'white',
                    'volume': 0.1,
                    'filter_type': 'highpass',
                    'filter_freq': 2000,
                    'spatial_width': 3.0,
                    'position': [0, 3, 0]  # Directly above
                })
            elif weather_type == 'storm':
                texture_config.update({
                    'type': 'brown',
                    'volume': 0.15,
                    'filter_type': 'lowpass',
                    'filter_freq': 500,
                    'spatial_width': 4.0,
                    'position': [0, 1, -2]  # Lower and further back
                })
            elif weather_type == 'wind':
                texture_config.update({
                    'type': 'pink',
                    'volume': 0.08 + (wind_speed / 100 * 0.1),
                    'filter_type': 'bandpass',
                    'filter_freq': 1000 + (wind_speed * 10),
                    'filter_q': 0.5,
                    'movement_pattern': {
                        'type': 'circular',
                        'radius': 2.0,
                        'speed': wind_speed / 20
                    }
                })
            elif weather_type == 'snow':
                texture_config.update({
                    'type': 'white',
                    'volume': 0.03,
                    'filter_type': 'lowpass',
                    'filter_freq': 3000,
                    'spatial_width': 5.0,
                    'position': [0, 4, 0]  # High above for falling snow feel
                })
            
            return self.create_noise(texture_config)
            
        except Exception as e:
            logger.error(f"Error creating ambient texture: {e}")
            return None
    
    def create_vinyl_noise(self, amount: float, weather_type: str = 'clear') -> Dict[str, Any]:
        """
        Create vinyl noise effect for lo-fi generator.
        
        Args:
            amount: Vinyl noise amount (0-1)
            weather_type: Weather type for specific characteristics
            
        Returns:
            Dictionary with vinyl noise configuration
        """
        try:
            # Base vinyl configuration
            vinyl_config = {
                'type': 'pink',  # Pink noise is closest to vinyl
                'volume': 0.025 * amount,
                'duration': 3.0,
                'loop': True,
                'position': [0, 0, 0],  # Center position
                'spatial_width': 0.5  # Narrow for vinyl feel
            }
            
            # Weather-specific vinyl characteristics
            if weather_type == 'rain':
                vinyl_config.update({
                    'filter_type': 'lowpass',
                    'filter_freq': 4000,
                    'volume': 0.03 * amount
                })
            elif weather_type == 'storm':
                vinyl_config.update({
                    'type': 'brown',  # Heavier noise for storms
                    'filter_type': 'lowpass',
                    'filter_freq': 3000,
                    'volume': 0.04 * amount
                })
            elif weather_type == 'night':
                vinyl_config.update({
                    'filter_type': 'lowpass',
                    'filter_freq': 5000,
                    'volume': 0.02 * amount
                })
            elif weather_type == 'clear':
                vinyl_config.update({
                    'filter_type': 'highpass',
                    'filter_freq': 300,
                    'volume': 0.015 * amount
                })
            
            return self.create_noise(vinyl_config)
            
        except Exception as e:
            logger.error(f"Error creating vinyl noise: {e}")
            return None
    
    def create_weather_noise_layer(self, weather_data: Dict[str, Any], 
                                  layer_type: str = 'background') -> Dict[str, Any]:
        """
        Create a noise layer appropriate for weather conditions.
        
        Args:
            weather_data: Weather conditions
            layer_type: Type of layer ('background', 'texture', 'accent')
            
        Returns:
            Dictionary with noise layer configuration
        """
        try:
            # Base configuration
            config = {
                'duration': 4.0,
                'loop': True
            }
            
            # Configure based on layer type
            if layer_type == 'background':
                config.update({
                    'type': 'brown',
                    'volume': 0.02,
                    'position': [0, 0, -3],
                    'spatial_width': 3.0
                })
            elif layer_type == 'texture':
                config.update({
                    'type': 'pink',
                    'volume': 0.05,
                    'position': [0, 1, -1],
                    'spatial_width': 2.0
                })
            elif layer_type == 'accent':
                config.update({
                    'type': 'white',
                    'volume': 0.03,
                    'position': [1, 2, 0],
                    'spatial_width': 1.0
                })
            
            # Adjust for weather conditions
            weather_type = weather_data.get('type', 'clear')
            intensity = weather_data.get('intensity', 0.5)
            
            if weather_type in ['rain', 'storm']:
                config['volume'] *= (1 + intensity)
                config['filter_type'] = 'highpass'
                config['filter_freq'] = 1000 * (1 + intensity)
            elif weather_type == 'wind':
                config['volume'] *= (0.5 + intensity)
                config['movement_pattern'] = {
                    'type': 'linear',
                    'direction': [1, 0, 0],
                    'speed': intensity
                }
            
            return self.create_noise(config)
            
        except Exception as e:
            logger.error(f"Error creating weather noise layer: {e}")
            return None


def create_spatial_noise_generator() -> SpatialNoise:
    """Factory function to create a spatial noise generator."""
    return SpatialNoise()