"""
SHAC Export Module for Atabey Symphony

This module provides convenient functions to export audio from Atabey's
sound generation pipeline directly to SHAC format files.
"""

import numpy as np
import logging
from typing import Dict, Optional, Union, List
from pathlib import Path

from .shac.codec import SHACCodec, SHACFileWriter
from .shac.codec.math_utils import AmbisonicNormalization
from .shac.codec.encoders import encode_mono_source, convert_to_spherical

# Set up logger
logger = logging.getLogger(__name__)


def save_audio_as_shac(
    audio_data: Union[np.ndarray, Dict[str, np.ndarray]], 
    filename: str,
    positions: Optional[Dict[str, tuple]] = None,
    metadata: Optional[Dict] = None,
    order: int = 3,
    sample_rate: int = 48000,
    room_dimensions: tuple = (10, 5, 10),
    distance_gain: bool = False
) -> str:
    """
    Save audio data as a SHAC file with spatial positioning.
    
    Args:
        audio_data: Either a mono/stereo array or dict of audio layers
        filename: Output filename (will add .shac extension if missing)
        positions: Dict mapping layer names to (x, y, z) positions
        metadata: Additional metadata to store
        order: Ambisonic order (default 3)
        sample_rate: Sample rate in Hz
        room_dimensions: Room size for acoustics (width, height, depth)
        
    Returns:
        Path to the saved SHAC file
    """
    # Ensure filename has .shac extension
    if not filename.endswith('.shac'):
        filename += '.shac'
    
    # Initialize SHAC codec
    codec = SHACCodec(
        order=order,
        sample_rate=sample_rate
    )
    
    # Handle different input formats
    if isinstance(audio_data, np.ndarray):
        # Single audio source
        audio_dict = {'main': audio_data}
        if positions is None:
            positions = {'main': (0, 0, 0)}
    else:
        # Multiple audio sources
        audio_dict = audio_data
        if positions is None:
            # Default positions in a circle
            positions = {}
            n_sources = len(audio_dict)
            for i, layer_name in enumerate(audio_dict.keys()):
                angle = 2 * np.pi * i / n_sources
                x = 3 * np.cos(angle)
                z = 3 * np.sin(angle)
                positions[layer_name] = (x, 0, z)
    
    # Create SHAC file writer
    writer = SHACFileWriter(
        order=order,
        sample_rate=sample_rate,
        normalization=AmbisonicNormalization.SN3D
    )
    
    # Process each audio layer
    for layer_name, audio in audio_dict.items():
        logger.info(f"Processing layer '{layer_name}': shape={audio.shape}, duration={len(audio)/sample_rate:.2f}s")
        
        # Handle multi-channel audio - preserve or intelligently process
        if audio.ndim > 1:
            if audio.shape[1] == 1:
                # Already mono, just reshape
                audio = audio.flatten()
                logger.debug(f"Preserved mono: shape={audio.shape}")
            elif audio.shape[1] == 2:
                # Stereo - user should choose: preserve left, preserve right, or mix
                # For now, preserve left channel to avoid forced mixing
                audio = audio[:, 0]
                logger.debug(f"Preserved left channel from stereo: shape={audio.shape}")
            else:
                # Multi-channel - preserve first channel
                channel_count = audio.shape[1]
                audio = audio[:, 0]
                logger.debug(f"Preserved first channel from {channel_count}-channel audio: shape={audio.shape}")
        else:
            logger.debug(f"Already mono: shape={audio.shape}")
        
        # Get position for this layer
        position = positions.get(layer_name, (0, 0, 0))
        logger.debug(f"Position: {position}")
        
        # Convert cartesian to spherical coordinates
        spherical_pos = convert_to_spherical(position)
        
        # Encode to ambisonics
        ambisonic_audio = encode_mono_source(audio, spherical_pos, order, 
                                           apply_distance_gain=distance_gain)
        
        # Prepare layer metadata
        layer_meta = {
            'position': position,
            'type': layer_name,
            'duration': len(audio) / sample_rate
        }
        
        if metadata:
            layer_meta.update(metadata)
        
        # Add layer to SHAC file
        writer.add_layer(layer_name, ambisonic_audio, layer_meta)
    
    # Write the file
    writer.write_file(filename)
    
    return filename


def export_weather_soundscape_as_shac(
    weather_data: Dict,
    duration: float,
    output_path: str,
    scene_config: Optional[Dict] = None
) -> str:
    """
    Export a weather soundscape directly to SHAC format.
    
    Args:
        weather_data: Weather parameters
        duration: Duration in seconds
        output_path: Output file path
        scene_config: Optional spatial scene configuration
        
    Returns:
        Path to saved SHAC file
    """
    from .weather_sound_bridge import create_weather_soundscape
    from .sound_generator_mid import UnifiedSoundGenerator
    
    # Generate weather soundscape components using the working generator
    generator = UnifiedSoundGenerator()
    
    # Create basic weather-responsive audio (simplified for now)
    audio_layers = {}
    positions = {}
    
    # Base ambient layer at center
    from .sound_engine import generate_sine, apply_envelope
    base_audio = generate_sine(220, duration, sample_rate=48000)
    envelope = np.linspace(1, 0.8, len(base_audio))
    audio_layers['ambient'] = apply_envelope(base_audio, envelope) * 0.3
    positions['ambient'] = (0, 0, 0)
    
    # Weather-specific positioning
    weather_type = weather_data.get('weather', 'clear').lower()
    
    if 'rain' in weather_type or 'storm' in weather_type:
        positions['ambient'] = (0, 1, 0)  # Slightly elevated for weather
    
    # Export as SHAC
    metadata = {
        'weather_data': weather_data,
        'generated_by': 'Atabey Symphony Weather System'
    }
    
    return save_audio_as_shac(
        audio_layers,
        output_path,
        positions=positions,
        metadata=metadata
    )